const mysql = require('mysql2/promise');
const bcrypt = require('bcrypt');
require('dotenv').config();

async function setupDatabase() {
  console.log('Starting database setup...');
  console.log(`Database: ${process.env.DB_NAME}`);
  console.log(`User: ${process.env.DB_USER}`);
  console.log(`Host: ${process.env.DB_HOST}`);
  
  let connection;

  try {
    // Connect to MySQL with database
    connection = await mysql.createConnection({
      host: process.env.DB_HOST,
      user: process.env.DB_USER,
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME
    });

    console.log('✅ Connected to MySQL database');

    // Create users table
    await connection.query(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(50) NOT NULL UNIQUE,
        password VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);
    console.log('✅ Users table created or verified');

    // Create site_config table
    await connection.query(`
      CREATE TABLE IF NOT EXISTS site_config (
        id INT AUTO_INCREMENT PRIMARY KEY,
        config_key VARCHAR(50) NOT NULL UNIQUE,
        config_value JSON NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);
    console.log('✅ Site_config table created or verified');

    // Check if default admin user exists
    const [users] = await connection.query('SELECT * FROM users WHERE username = ?', ['admin']);

    if (users.length === 0) {
      // Create default admin user
      const hashedPassword = await bcrypt.hash('cameoadmin123', 10);
      await connection.query(
        'INSERT INTO users (username, password) VALUES (?, ?)',
        ['admin', hashedPassword]
      );
      console.log('✅ Default admin user created');
    } else {
      console.log('ℹ️ Admin user already exists');
    }

    // Check if site config already exists in database
    const [configExists] = await connection.query('SELECT * FROM site_config WHERE config_key = ?', ['main']);

    if (configExists.length === 0) {
      // Create default site configuration
      const defaultConfig = {
        profile: {
          name: "Cameo",
          subtitle: "Musician & Songwriter",
          imagePath: "/assets/android-chrome-512x512.png"
        },
        links: {
          spotify: "#",
          apple: "#",
          youtube: "#",
          youtubeMusic: "#",
          amazon: "#"
        },
        video: {
          title: "Latest Release",
          videoId: "dQw4w9WgXcQ"
        },
        social: {
          instagram: "#",
          facebook: "#",
          tiktok: "#"
        },
        bio: "Cameo is a versatile musician and producer from Moldova who creates innovative sounds across multiple genres. Drawing inspiration from Western culture while blending Moldavian, Romanian and Russian styles, Cameo has developed a unique sound that resonates with audiences from various countries."
      };

      await connection.query(
        'INSERT INTO site_config (config_key, config_value) VALUES (?, ?)',
        ['main', JSON.stringify(defaultConfig)]
      );
      console.log('✅ Default site configuration created');
    } else {
      console.log('ℹ️ Site configuration already exists');
    }

    console.log('✅ Database setup completed successfully');
  } catch (error) {
    console.error('❌ Database setup failed:', error);
  } finally {
    if (connection) {
      await connection.end();
      console.log('ℹ️ Database connection closed');
    }
  }
}

// Run the setup
setupDatabase()
  .then(() => {
    console.log('Setup script finished.');
    // Exit with success code
    process.exit(0);
  })
  .catch(err => {
    console.error('Fatal error:', err);
    // Exit with error code
    process.exit(1);
  });
