const mysql = require('mysql2/promise');
const bcrypt = require('bcrypt');
require('dotenv').config();

async function setupDatabase() {
  let connection;

  try {
    // Connect to MySQL without database (to create it if needed)
    connection = await mysql.createConnection({
      host: process.env.DB_HOST,
      user: process.env.DB_USER,
      password: process.env.DB_PASSWORD
    });

    console.log('Connected to MySQL server');

    // Create database if it doesn't exist
    await connection.query(`CREATE DATABASE IF NOT EXISTS ${process.env.DB_NAME}`);
    console.log(`Database ${process.env.DB_NAME} created or verified`);

    // Use the database
    await connection.query(`USE ${process.env.DB_NAME}`);

    // Create users table
    await connection.query(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(50) NOT NULL UNIQUE,
        password VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);
    console.log('Users table created or verified');

    // Create site_config table
    await connection.query(`
      CREATE TABLE IF NOT EXISTS site_config (
        id INT AUTO_INCREMENT PRIMARY KEY,
        config_key VARCHAR(50) NOT NULL UNIQUE,
        config_value JSON NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);
    console.log('Site_config table created or verified');

    // Check if default admin user exists
    const [users] = await connection.query('SELECT * FROM users WHERE username = ?', ['admin']);

    if (users.length === 0) {
      // Create default admin user
      const hashedPassword = await bcrypt.hash('cameoadmin123', 10);
      await connection.query(
        'INSERT INTO users (username, password) VALUES (?, ?)',
        ['admin', hashedPassword]
      );
      console.log('Default admin user created');
    } else {
      console.log('Admin user already exists');
    }

    // Import existing site configuration from JSON if it exists
    const fs = require('fs');
    const path = require('path');
    const siteConfigPath = path.join(__dirname, 'data/site-config.json');

    if (fs.existsSync(siteConfigPath)) {
      const siteConfig = JSON.parse(fs.readFileSync(siteConfigPath));

      // Check if site config already exists in database
      const [configExists] = await connection.query('SELECT * FROM site_config WHERE config_key = ?', ['main']);

      if (configExists.length === 0) {
        // Insert site configuration
        await connection.query(
          'INSERT INTO site_config (config_key, config_value) VALUES (?, ?)',
          ['main', JSON.stringify(siteConfig)]
        );
        console.log('Site configuration imported from JSON file');
      } else {
        console.log('Site configuration already exists in database');
      }
    }

    console.log('Database setup completed successfully');
  } catch (error) {
    console.error('Database setup failed:', error);
  } finally {
    if (connection) {
      await connection.end();
      console.log('Database connection closed');
    }
  }
}

setupDatabase();
