const express = require('express');
const path = require('path');
const fs = require('fs');
const app = express();
const jwt = require('jsonwebtoken');
const multer = require('multer');
const bodyParser = require('body-parser');
const bcrypt = require('bcrypt');
const compression = require('compression'); // Import compression middleware
require('dotenv').config();

// Import database configuration
const { pool, testConnection } = require('./config/database');

// Update the PORT variable to ensure it works on your hosting platform
const PORT = process.env.PORT || 3000;

// Configure compression - add before other middleware
app.use(compression({
  // Compression level (0-9): higher means more compression but slower
  level: 6,
  // Only compress responses larger than 1KB
  threshold: 1024,
  // Don't compress responses with these content types
  filter: (req, res) => {
    if (req.headers['x-no-compression']) {
      // Don't compress responses with this request header
      return false;
    }
    // Compress all other responses
    return compression.filter(req, res);
  }
}));

// Security headers middleware
app.use((req, res, next) => {
  // HTTP Strict Transport Security
  res.setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
  
  // Content Security Policy - Updated to allow Google Analytics
  res.setHeader('Content-Security-Policy', "default-src 'self'; script-src 'self' https://www.youtube.com https://s.ytimg.com https://www.google-analytics.com https://www.googletagmanager.com https://cdnjs.cloudflare.com https://*.doubleclick.net https://stats.g.doubleclick.net https://region1.google-analytics.com 'unsafe-inline'; frame-src https://www.youtube.com; connect-src 'self' https://www.youtube.com https://*.doubleclick.net https://*.googlesyndication.com https://*.g.doubleclick.net https://stats.g.doubleclick.net https://region1.google-analytics.com https://*.google-analytics.com https://analytics.google.com; style-src 'self' https://fonts.googleapis.com https://cdnjs.cloudflare.com https://fonts.cdnfonts.com 'unsafe-inline'; font-src 'self' https://fonts.gstatic.com https://cdnjs.cloudflare.com https://fonts.cdnfonts.com; img-src 'self' data: https://i.ytimg.com https://www.google-analytics.com https://*.doubleclick.net https://stats.g.doubleclick.net");
  
  // Prevent MIME type sniffing
  res.setHeader('X-Content-Type-Options', 'nosniff');
  
  // Prevent clickjacking
  res.setHeader('X-Frame-Options', 'DENY');
  
  // XSS Protection
  res.setHeader('X-XSS-Protection', '1; mode=block');
  
  // Don't send referrer information to other sites
  res.setHeader('Referrer-Policy', 'same-origin');
  
  // Add no-cache headers for video resources
  if (req.url.includes('video') || req.url.includes('youtube')) {
    res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
    res.setHeader('Pragma', 'no-cache');
    res.setHeader('Expires', '0');
  }
  
  next();
});

// Configure file uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    // Different destinations for different file types
    if (file.mimetype.startsWith('image/')) {
      cb(null, 'public/assets/uploads/images/')
    } else if (file.mimetype.startsWith('video/')) {
      cb(null, 'public/assets/uploads/videos/')
    } else {
      cb(null, 'public/assets/uploads/')
    }
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const ext = path.extname(file.originalname);
    
    if (file.mimetype.startsWith('image/')) {
      cb(null, 'profile-' + uniqueSuffix + ext);
    } else if (file.mimetype.startsWith('video/')) {
      cb(null, 'background-' + uniqueSuffix + ext);
    } else {
      cb(null, file.fieldname + '-' + uniqueSuffix + ext);
    }
  }
});

const upload = multer({
  storage: storage,
  limits: { 
    fileSize: 52428800 // 50MB limit
  },
  fileFilter: (req, file, cb) => {
    // Allow images and videos
    if (file.mimetype.startsWith('image/') || 
        file.mimetype.startsWith('video/') && 
        (file.mimetype === 'video/mp4' || file.mimetype === 'video/webm' || file.mimetype === 'video/ogg')) {
      cb(null, true);
    } else {
      cb(new Error('Only images and video files are allowed'));
    }
  }
});

// Middleware for parsing JSON data with larger size limit
app.use(bodyParser.json({ limit: '2mb' }));

// Add special no-cache middleware for admin routes
app.use(['/admin', '/login'], (req, res, next) => {
  res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
  res.setHeader('Pragma', 'no-cache');
  res.setHeader('Expires', '0');
  res.setHeader('Surrogate-Control', 'no-store');
  next();
});

app.use(express.static(path.join(__dirname, 'public'), { 
  maxAge: 86400000, // 1 day cache for regular static assets
  setHeaders: (res, path) => {
    // Don't cache HTML files
    if (path.endsWith('.html')) {
      res.setHeader('Cache-Control', 'no-cache, no-store, must-revalidate');
      res.setHeader('Pragma', 'no-cache');
      res.setHeader('Expires', '0');
    }
    // Don't cache JS files that control dynamic content or CSS files
    if (path.endsWith('main.js') || path.endsWith('admin.js') || path.endsWith('.css') ||
        path.endsWith('main.js?v=2.0') || path.endsWith('video-background.js?v=2.0') ||
        path.endsWith('styles.css?v=2.0') || path.endsWith('video-background.css?v=2.0')) {
      res.setHeader('Cache-Control', 'no-cache, no-store, must-revalidate');
      res.setHeader('Pragma', 'no-cache');
      res.setHeader('Expires', '0');
    }
  }
}));

// Create directories if they don't exist
const uploadsDir = path.join(__dirname, 'public/assets/uploads');
const imageUploadsDir = path.join(__dirname, 'public/assets/uploads/images');
const videoUploadsDir = path.join(__dirname, 'public/assets/uploads/videos');

if (!fs.existsSync(uploadsDir)) {
  fs.mkdirSync(uploadsDir, { recursive: true });
}
if (!fs.existsSync(imageUploadsDir)) {
  fs.mkdirSync(imageUploadsDir, { recursive: true });
}
if (!fs.existsSync(videoUploadsDir)) {
  fs.mkdirSync(videoUploadsDir, { recursive: true });
}

// Rate limiting for API endpoints (simple implementation)
const apiRequestCounts = {};
const RATE_LIMIT_WINDOW = 60 * 1000; // 1 minute
const RATE_LIMIT_MAX = 100; // max requests per window

function rateLimiter(req, res, next) {
  const ip = req.ip;
  const now = Date.now();
  
  if (!apiRequestCounts[ip]) {
    apiRequestCounts[ip] = {
      count: 1,
      resetTime: now + RATE_LIMIT_WINDOW
    };
    return next();
  }
  
  if (now > apiRequestCounts[ip].resetTime) {
    apiRequestCounts[ip] = {
      count: 1,
      resetTime: now + RATE_LIMIT_WINDOW
    };
    return next();
  }
  
  if (apiRequestCounts[ip].count >= RATE_LIMIT_MAX) {
    return res.status(429).json({ 
      message: 'Too many requests, please try again later.' 
    });
  }
  
  apiRequestCounts[ip].count++;
  next();
}

// Apply rate limiting to API routes
app.use('/api', rateLimiter);

// Global cache control middleware for API endpoints
app.use('/api', (req, res, next) => {
  // Strong no-cache headers for all API responses
  res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
  res.setHeader('Pragma', 'no-cache');
  res.setHeader('Expires', '0');
  res.setHeader('Surrogate-Control', 'no-store');
  // Add a random ETag to ensure browser never uses cached response
  res.setHeader('ETag', Date.now().toString() + Math.random());
  next();
});

// Authentication middleware
async function authenticateToken(req, res, next) {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];
  
  if (token == null) return res.status(401).json({ message: 'Authentication required' });
  
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (err) {
    return res.status(403).json({ message: 'Invalid or expired token' });
  }
}

// Serve the main page with dynamic data and SEO
app.get('/', async (req, res) => {
  try {
    // Get the site configuration
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length > 0) {
      const config = JSON.parse(rows[0].config_value);
      
      // Read the HTML file
      let html = fs.readFileSync(path.join(__dirname, 'public', 'index.html'), 'utf8');
      
      // Replace meta tags if they exist in the config
      if (config.seo) {
        // Update meta title
        if (config.seo.metaTitle) {
          html = html.replace(/<title>[^<]*<\/title>/i, `<title>${config.seo.metaTitle}</title>`);
        }
        
        // Update website title (h1 element)
        if (config.seo.websiteTitle) {
          html = html.replace(/<h1 class="natural-precision">[^<]*<\/h1>/i, 
            `<h1 class="natural-precision">${config.seo.websiteTitle}</h1>`);
        }
        
        // Update meta description
        if (config.seo.metaDescription) {
          html = html.replace(/<meta\s+name="description"\s+content="[^"]*"/i, 
            `<meta name="description" content="${config.seo.metaDescription}"`);
        }
        
        // Insert Google Analytics code if available - FIXED VERSION
        if (config.seo.gaTrackingCode && config.seo.gaTrackingCode.trim() !== '') {
          console.log("Applying Google Analytics code to page");
          
          // Clean the tracking code but preserve script functionality
          let cleanTrackingCode = config.seo.gaTrackingCode
            .replace(/<!--[\s\S]*?-->/g, '')
            .trim();
          
          // Try to find the placeholder script tag with better regex
          if (html.includes('<script id="google-analytics">')) {
            // Replace the placeholder with the actual tracking code
            html = html.replace(
              /<script id="google-analytics">[\s\S]*?<\/script>/i, 
              cleanTrackingCode
            );
            console.log("Google Analytics placeholder replaced");
          } else {
            // If placeholder wasn't found, insert just before </body>
            html = html.replace('</body>', cleanTrackingCode + '\n</body>');
            console.log("Google Analytics added before </body>");
          }
        } else {
          console.log("No Google Analytics tracking code found in config");
        }
      }
      
      // Send the modified HTML
      res.send(html);
    } else {
      // Fall back to static file if no config
      res.sendFile(path.join(__dirname, 'public', 'index.html'));
    }
  } catch (error) {
    console.error('Error serving index with SEO:', error);
    // Fall back to static file on error
    res.sendFile(path.join(__dirname, 'public', 'index.html'));
  }
});

// Dynamic sitemap generation route - place this before the static file middleware
app.get('/sitemap.xml', async (req, res) => {
  try {
    // Get the site configuration from database
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length === 0) {
      // Fallback to static sitemap if no config found
      return res.sendFile(path.join(__dirname, 'public', 'sitemap.xml'));
    }
    
    // Parse the configuration
    const config = JSON.parse(rows[0].config_value);
    
    // Format current date as YYYY-MM-DD for lastmod
    const today = new Date().toISOString().split('T')[0];
    
    // Get artist information
    const artistName = config.profile?.name || 'Cameo';
    
    // Generate simplified sitemap XML content without the video tag
    const sitemapXml = `<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
  <url>
    <loc>https://cameomusic.art/</loc>
    <lastmod>${today}</lastmod>
    <changefreq>weekly</changefreq>
    <priority>1.0</priority>
  </url>
</urlset>`;
    
    // Send the XML response
    res.header('Content-Type', 'application/xml');
    res.send(sitemapXml);
    
  } catch (error) {
    console.error('Error generating sitemap:', error);
    // Fallback to static sitemap on error
    res.sendFile(path.join(__dirname, 'public', 'sitemap.xml'));
  }
});

// Serve login page
app.get('/login', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'login.html'));
});

// Serve the admin page - improved token handling
app.get('/admin', (req, res) => {
  // First check for token in query parameter (from login redirect)
  const queryToken = req.query.token;
  
  // Then check for token in Authorization header (for API calls)
  const authHeader = req.headers['authorization'];
  const headerToken = authHeader && authHeader.split(' ')[1];
  
  // Also check cookies
  const cookies = req.headers.cookie;
  let cookieToken = null;
  if (cookies) {
    const tokenCookie = cookies.split(';').find(c => c.trim().startsWith('adminToken='));
    if (tokenCookie) {
      cookieToken = tokenCookie.split('=')[1];
    }
  }
  
  // Use any available token
  const accessToken = queryToken || headerToken || cookieToken;
  
  if (!accessToken) {
    console.log('No access token found, redirecting to login');
    return res.redirect('/login');
  }
  
  try {
    // Verify the token
    const decoded = jwt.verify(accessToken, process.env.JWT_SECRET);
    
    // Set strongest no-cache headers for admin page
    res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
    res.setHeader('Pragma', 'no-cache');
    res.setHeader('Expires', '0');
    res.setHeader('Surrogate-Control', 'no-store');
    
    // If we got here with a query token, set a cookie for future requests
    if (queryToken) {
      res.cookie('adminToken', queryToken, {
        httpOnly: true,
        maxAge: 7200000, // 2 hours
        sameSite: 'strict',
        secure: process.env.NODE_ENV === 'production' // secure in production
      });
    }
    
    // Valid token, serve admin page
    res.sendFile(path.join(__dirname, 'public', 'admin.html'));
  } catch (error) {
    console.log('Invalid token:', error.message);
    // Set a flag to indicate token verification failed
    return res.redirect('/login?auth_error=token_invalid');
  }
});

// API endpoint to get site configuration - improved cache headers
app.get('/api/site-config', async (req, res) => {
  try {
    const [rows] = await pool.query('SELECT config_value, updated_at FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length > 0) {
      const config = JSON.parse(rows[0].config_value);
      
      // Add timestamp for client-side change detection
      config._lastUpdated = rows[0].updated_at.toString();
      config._serverTime = new Date().getTime();
      // Add unique identifier to help client detect changes
      config._cacheBuster = Date.now().toString() + Math.random().toString(36).substring(2, 10);
      
      res.json(config);
    } else {
      res.status(404).json({ message: 'Site configuration not found' });
    }
  } catch (error) {
    console.error('Error loading site configuration:', error);
    res.status(500).json({ message: 'Error loading site configuration' });
  }
});

// Admin login endpoint - using MySQL database
app.post('/api/admin/login', async (req, res) => {
  try {
    const { username, password } = req.body;
    
    // Input validation
    if (!username || !password) {
      return res.status(400).json({ message: 'Username and password are required' });
    }
    
    console.log('Login attempt for user:', username);
    
    // Check user in database
    const [users] = await pool.query('SELECT * FROM users WHERE username = ?', [username]);
    
    if (users.length === 0) {
      console.log('Login failed: User not found');
      return res.status(401).json({ message: 'Invalid username or password' });
    }
    
    const user = users[0];
    
    // Verify password
    const passwordMatch = await bcrypt.compare(password, user.password);
    
    if (!passwordMatch) {
      console.log('Login failed: Invalid password for user:', username);
      return res.status(401).json({ message: 'Invalid username or password' });
    }
    
    // Generate JWT token
    const token = jwt.sign({ id: user.id, username: user.username }, process.env.JWT_SECRET, { expiresIn: '2h' });
    
    // Set HTTP-only cookie with token
    res.cookie('adminToken', token, {
      httpOnly: true,
      maxAge: 7200000, // 2 hours
      sameSite: 'strict',
      secure: process.env.NODE_ENV === 'production' // secure in production
    });
    
    console.log('Login successful for user:', username);
    return res.json({ success: true, token });
    
  } catch (error) {
    console.error('Login error:', error);
    return res.status(500).json({ message: 'Server error during login' });
  }
});

// Verify token endpoint - improved with better error handling
app.get('/api/admin/verify', (req, res) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];
  
  if (!token) {
    return res.status(401).json({ valid: false, message: 'No token provided' });
  }
  
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    res.json({ valid: true, user: decoded.username });
  } catch (err) {
    console.log('Token verification failed:', err.message);
    res.status(401).json({ valid: false, message: err.message });
  }
});

// Get site data for admin
app.get('/api/admin/site-data', authenticateToken, async (req, res) => {
  try {
    // Add extra cache control for this endpoint
    res.setHeader('Cache-Control', 'no-store, must-revalidate');
    
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length > 0) {
      const data = JSON.parse(rows[0].config_value);
      // Add timestamp to help with client-side cache detection
      data._fetchTime = Date.now();
      res.json(data);
    } else {
      res.status(404).json({ message: 'Site configuration not found' });
    }
  } catch (error) {
    console.error('Error loading site data:', error);
    res.status(500).json({ message: 'Error loading site configuration' });
  }
});

// Save site data with improved cache invalidation
app.post('/api/admin/save-site-data', authenticateToken, async (req, res) => {
  try {
    let configData = req.body;
    
    // Make validation more flexible - check for object existence but don't require all fields
    if (!configData || typeof configData !== 'object') {
      return res.status(400).json({ message: 'Invalid data format' });
    }
    
    // Ensure minimum required structure exists, but allow empty objects
    configData.profile = configData.profile || {};
    configData.links = configData.links || {};
    configData.video = configData.video || {};
    configData.video.noCache = true; // Always enforce no-cache for featured video
    configData.social = configData.social || {};
    configData.seo = configData.seo || {}; // Ensure seo section exists
    
    // Add or update background video no-cache setting
    if (configData.backgroundVideo) {
      configData.backgroundVideo.noCache = true;
    }
    
    // Force the cache settings to disable caching
    configData.cacheSettings = {
      cacheBackgroundVideo: false,
      lastUpdated: new Date().toISOString()
    };
    
    // Check if config exists
    const [rows] = await pool.query('SELECT * FROM site_config WHERE config_key = ?', ['main']);
    
    // If existing config exists, merge with new data to preserve omitted fields
    if (rows.length > 0) {
      const existingConfig = JSON.parse(rows[0].config_value);
      
      // Deep merge to ensure we don't lose nested properties that aren't included in the update
      configData = {
        ...existingConfig,
        profile: { ...(existingConfig.profile || {}), ...(configData.profile || {}) },
        links: { ...(existingConfig.links || {}), ...(configData.links || {}) },
        countdown: { 
          ...(existingConfig.countdown || {}), 
          ...(configData.countdown || {}),
          // If enabled is undefined but showTimer is true, make sure we enable the section
          enabled: configData.countdown?.enabled !== undefined 
            ? configData.countdown.enabled 
            : (configData.countdown?.showTimer === true ? true : existingConfig.countdown?.enabled)
        },
        video: { ...(existingConfig.video || {}), ...(configData.video || {}) },
        social: { ...(existingConfig.social || {}), ...(configData.social || {}) },
        seo: { ...(existingConfig.seo || {}), ...(configData.seo || {}) },
        // Always take the new bio if provided
        bio: configData.bio !== undefined ? configData.bio : existingConfig.bio,
        // Always take the new background video if provided
        backgroundVideo: configData.backgroundVideo || existingConfig.backgroundVideo,
        // Always take the new cache settings if provided
        cacheSettings: configData.cacheSettings || existingConfig.cacheSettings
      };
      
      // If useDefaultImage is true or the SVG is specified, remove custom imagePath
      if (configData.profile && (configData.profile.useDefaultImage || 
          (configData.profile.imagePath && configData.profile.imagePath.includes('Round.svg')))) {
        delete configData.profile.imagePath;
      }
    }
    
    // Generate a unique cache busting ID
    const cacheBuster = Date.now().toString();
    
    // Add the cache buster to the data for client-side detection
    configData._cacheBuster = cacheBuster;
    
    if (rows.length > 0) {
      // Update existing config
      await pool.query(
        'UPDATE site_config SET config_value = ?, updated_at = CURRENT_TIMESTAMP WHERE config_key = ?',
        [JSON.stringify(configData), 'main']
      );
    } else {
      // Insert new config
      await pool.query(
        'INSERT INTO site_config (config_key, config_value) VALUES (?, ?)',
        ['main', JSON.stringify(configData)]
      );
    }
    
    // Also store the cache busting ID separately for quicker access
    await pool.query(
      'INSERT INTO site_config (config_key, config_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)',
      ['cache_version', JSON.stringify({ version: cacheBuster })]
    );
    
    // Add more detailed logging to help identify issues
    console.log(`Configuration saved successfully. Cache buster: ${cacheBuster}`);
    
    // Add update timestamp to the response to help with content refresh
    res.json({ 
      success: true, 
      updated: new Date().toISOString(),
      cacheBuster: cacheBuster
    });
  } catch (error) {
    console.error('Save error:', error);
    // Add more detailed error information in development mode
    const errorMsg = process.env.NODE_ENV === 'production' 
      ? 'Error saving configuration' 
      : `Error saving configuration: ${error.message}`;
    res.status(500).json({ message: errorMsg });
  }
});

// Add a lightweight endpoint to check for content changes
app.get('/api/cache-version', async (req, res) => {
  try {
    // Strong no-cache headers
    res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private');
    res.setHeader('Pragma', 'no-cache');
    res.setHeader('Expires', '0');
    
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['cache_version']);
    
    if (rows.length > 0) {
      res.json(JSON.parse(rows[0].config_value));
    } else {
      res.json({ version: '0' });
    }
  } catch (error) {
    console.error('Error checking cache version:', error);
    res.status(500).json({ version: '0' });
  }
});

// Upload image endpoint
app.post('/api/admin/upload-image', authenticateToken, upload.single('profileImage'), (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ message: 'No file uploaded' });
    }
    
    const imagePath = '/assets/uploads/' + req.file.filename;
    res.json({ success: true, imagePath });
  } catch (error) {
    console.error('Upload error:', error);
    res.status(500).json({ message: 'Error uploading image' });
  }
});

// Upload video endpoint
app.post('/api/admin/upload-video', authenticateToken, upload.single('backgroundVideo'), (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ message: 'No file uploaded' });
    }
    
    const videoPath = '/assets/uploads/videos/' + req.file.filename;
    res.json({ success: true, videoPath });
  } catch (error) {
    console.error('Upload error:', error);
    res.status(500).json({ message: 'Error uploading video' });
  }
});

// Upload countdown image endpoint
app.post('/api/admin/upload-countdown-image', authenticateToken, upload.single('countdownImage'), (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ message: 'No file uploaded' });
    }
    
    const imagePath = '/assets/uploads/images/' + req.file.filename;
    res.json({ success: true, imagePath });
  } catch (error) {
    console.error('Upload error:', error);
    res.status(500).json({ message: 'Error uploading countdown image' });
  }
});

// Remove countdown image endpoint
app.post('/api/admin/remove-countdown-image', authenticateToken, async (req, res) => {
  try {
    // Get current site configuration
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Site configuration not found' });
    }
    
    const config = JSON.parse(rows[0].config_value);
    
    // If there was a custom image path, we need to check if we should delete the file
    if (config.countdown && config.countdown.imagePath) {
      try {
        // Check if file exists
        const imagePath = path.join(__dirname, 'public', config.countdown.imagePath);
        if (fs.existsSync(imagePath)) {
          // Only delete if it's in the uploads directory (security)
          if (imagePath.includes('uploads')) {
            fs.unlinkSync(imagePath);
            console.log('Removed countdown image file:', imagePath);
          }
        }
      } catch (fileError) {
        console.error('Error deleting countdown image file:', fileError);
        // Continue even if file deletion fails
      }
      
      // Remove the image path from configuration
      delete config.countdown.imagePath;
      
      // Update the configuration in database
      await pool.query(
        'UPDATE site_config SET config_value = ?, updated_at = CURRENT_TIMESTAMP WHERE config_key = ?',
        [JSON.stringify(config), 'main']
      );
      
      // Update cache version for change detection
      const cacheBuster = Date.now().toString();
      await pool.query(
        'INSERT INTO site_config (config_key, config_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)',
        ['cache_version', JSON.stringify({ version: cacheBuster })]
      );
      
      return res.json({ 
        success: true, 
        message: 'Countdown image removed successfully',
        cacheBuster
      });
    } else {
      // No custom image was set
      return res.json({ 
        success: true, 
        message: 'No countdown image to remove'
      });
    }
  } catch (error) {
    console.error('Error removing countdown image:', error);
    return res.status(500).json({ message: 'Server error removing countdown image' });
  }
});

// Remove profile image endpoint
app.post('/api/admin/remove-image', authenticateToken, async (req, res) => {
  try {
    // Get current site configuration
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Site configuration not found' });
    }
    
    const config = JSON.parse(rows[0].config_value);
    
    // If there was a custom image path, we need to check if we should delete the file
    if (config.profile && config.profile.imagePath) {
      try {
        // Check if file exists
        const imagePath = path.join(__dirname, 'public', config.profile.imagePath);
        if (fs.existsSync(imagePath)) {
          // Only delete if it's in the uploads directory (security)
          if (imagePath.includes('uploads')) {
            fs.unlinkSync(imagePath);
            console.log('Removed profile image file:', imagePath);
          }
        }
      } catch (fileError) {
        console.error('Error deleting profile image file:', fileError);
        // Continue even if file deletion fails
      }
      
      // Remove the image path from configuration
      delete config.profile.imagePath;
      
      // Update the configuration in database
      await pool.query(
        'UPDATE site_config SET config_value = ?, updated_at = CURRENT_TIMESTAMP WHERE config_key = ?',
        [JSON.stringify(config), 'main']
      );
      
      // Update cache version for change detection
      const cacheBuster = Date.now().toString();
      await pool.query(
        'INSERT INTO site_config (config_key, config_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)',
        ['cache_version', JSON.stringify({ version: cacheBuster })]
      );
      
      return res.json({ 
        success: true, 
        message: 'Profile image removed successfully',
        cacheBuster
      });
    } else {
      // No custom image was set
      return res.json({ 
        success: true, 
        message: 'No custom profile image to remove'
      });
    }
  } catch (error) {
    console.error('Error removing profile image:', error);
    return res.status(500).json({ message: 'Server error removing profile image' });
  }
});

// Remove video endpoint
app.post('/api/admin/remove-video', authenticateToken, async (req, res) => {
  try {
    // Get current site configuration
    const [rows] = await pool.query('SELECT config_value FROM site_config WHERE config_key = ?', ['main']);
    
    if (rows.length === 0) {
      return res.status(404).json({ message: 'Site configuration not found' });
    }
    
    const config = JSON.parse(rows[0].config_value);
    
    // If there was a custom video path, we need to check if we should delete the file
    if (config.backgroundVideo && config.backgroundVideo.videoPath) {
      try {
        // Check if file exists
        const videoPath = path.join(__dirname, 'public', config.backgroundVideo.videoPath);
        if (fs.existsSync(videoPath)) {
          // Only delete if it's in the videos directory (security)
          if (videoPath.includes('videos')) {
            fs.unlinkSync(videoPath);
            console.log('Removed background video file:', videoPath);
          }
        }
      } catch (fileError) {
        console.error('Error deleting background video file:', fileError);
        // Continue even if file deletion fails
      }
      
      // Remove the video path from configuration
      if (config.backgroundVideo) {
        delete config.backgroundVideo.videoPath;
      }
      
      // Update the configuration in database
      await pool.query(
        'UPDATE site_config SET config_value = ?, updated_at = CURRENT_TIMESTAMP WHERE config_key = ?',
        [JSON.stringify(config), 'main']
      );
      
      // Update cache version for change detection
      const cacheBuster = Date.now().toString();
      await pool.query(
        'INSERT INTO site_config (config_key, config_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)',
        ['cache_version', JSON.stringify({ version: cacheBuster })]
      );
      
      return res.json({ 
        success: true, 
        message: 'Background video removed successfully',
        cacheBuster
      });
    } else {
      // No custom video was set
      return res.json({ 
        success: true, 
        message: 'No custom background video to remove'
      });
    }
  } catch (error) {
    console.error('Error removing background video:', error);
    return res.status(500).json({ message: 'Server error removing background video' });
  }
});

// Update admin credentials
app.post('/api/admin/update-credentials', authenticateToken, async (req, res) => {
  try {
    const { username, password } = req.body;
    
    if (!username && !password) {
      return res.status(400).json({ message: 'No changes provided' });
    }
    
    const userId = req.user.id;
    
    // If changing username, check if the new username is already taken
    if (username) {
      const [existingUsers] = await pool.query(
        'SELECT * FROM users WHERE username = ? AND id != ?',
        [username, userId]
      );
      
      if (existingUsers.length > 0) {
        return res.status(400).json({ message: 'Username already taken' });
      }
    }
    
    // Update user based on what was provided
    if (username && password) {
      const hashedPassword = await bcrypt.hash(password, 10);
      await pool.query(
        'UPDATE users SET username = ?, password = ? WHERE id = ?',
        [username, hashedPassword, userId]
      );
    } else if (username) {
      await pool.query(
        'UPDATE users SET username = ? WHERE id = ?',
        [username, userId]
      );
    } else if (password) {
      const hashedPassword = await bcrypt.hash(password, 10);
      await pool.query(
        'UPDATE users SET password = ? WHERE id = ?',
        [hashedPassword, userId]
      );
    }
    
    res.json({ success: true });
  } catch (error) {
    console.error('Update credentials error:', error);
    res.status(500).json({ message: 'Error updating credentials' });
  }
});

// Spotify authentication route
app.get('/spotify-callback', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'spotify-callback.html'));
});

// Spotify API proxy endpoint
app.post('/api/spotify/token', async (req, res) => {
  try {
    const { code } = req.body;
    
    if (!code) {
      return res.status(400).json({ message: 'Authorization code required' });
    }
    
    // Exchange authorization code for access token
    const tokenResponse = await fetch('https://accounts.spotify.com/api/token', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'Authorization': `Basic ${Buffer.from(
          process.env.SPOTIFY_CLIENT_ID + ':' + process.env.SPOTIFY_CLIENT_SECRET
        ).toString('base64')}`
      },
      body: new URLSearchParams({
        grant_type: 'authorization_code',
        code,
        redirect_uri: `${req.protocol}://${req.get('host')}/spotify-callback`
      })
    });
    
    const tokenData = await tokenResponse.json();
    
    if (tokenResponse.ok) {
      res.json(tokenData);
    } else {
      res.status(tokenResponse.status).json(tokenData);
    }
  } catch (error) {
    console.error('Spotify token error:', error);
    res.status(500).json({ message: 'Error exchanging authorization code' });
  }
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('Server error:', err);
  res.status(500).json({
    message: 'Internal server error',
    error: process.env.NODE_ENV === 'production' ? {} : err.message
  });
});

// Start the server after testing database connection
async function startServer() {
  const dbConnected = await testConnection();
  if (dbConnected) {
    app.listen(PORT, () => {
      console.log(`Server running on http://localhost:${PORT}`);
      console.log(`Environment: ${process.env.NODE_ENV || 'development'}`);
    });
  } else {
    console.error('Server not started due to database connection failure');
  }
}

startServer();
