// Simple Service Worker for Cameo website

const CACHE_NAME = 'cameo-website-v2'; // Bump version to force update
const ASSETS_TO_CACHE = [
  '/',
  '/index.html',
  '/css/styles.css?v=2.0',
  '/css/video-background.css?v=2.0',
  '/js/main.js?v=2.0',
  '/js/video-background.js?v=2.0',
  '/assets/Round.svg',
  '/assets/favicon.ico',
  '/assets/apple-touch-icon.png',
  '/assets/favicon-32x32.png',
  '/assets/favicon-16x16.png',
  'https://fonts.googleapis.com/css2?family=Montserrat:wght@300;400;600&display=swap',
  'https://fonts.cdnfonts.com/css/natural-precision',
  'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css'
];

// Install event - cache assets
self.addEventListener('install', (event) => {
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then((cache) => {
        return cache.addAll(ASSETS_TO_CACHE);
      })
      .then(() => self.skipWaiting())
  );
});

// Activate event - clean up old caches
self.addEventListener('activate', (event) => {
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames
          .filter((cacheName) => cacheName !== CACHE_NAME)
          .map((cacheName) => caches.delete(cacheName))
      );
    })
  );
});

// Fetch event - serve from cache, fall back to network
self.addEventListener('fetch', (event) => {
  // Don't cache API requests
  if (event.request.url.includes('/api/')) {
    return;
  }

  // Always try network first for CSS/JS, fallback to cache
  if (event.request.url.match(/\.(css|js)(\?|$)/)) {
    event.respondWith(
      fetch(event.request)
        .then(response => {
          // Update cache
          if (response && response.status === 200) {
            const responseToCache = response.clone();
            caches.open(CACHE_NAME).then(cache => {
              cache.put(event.request, responseToCache);
            });
          }
          return response;
        })
        .catch(() => caches.match(event.request))
    );
    return;
  }

  // Default: cache first, then network
  event.respondWith(
    caches.match(event.request)
      .then((cachedResponse) => {
        if (cachedResponse) {
          return cachedResponse;
        }
        return fetch(event.request)
          .then((response) => {
            if (!response || response.status !== 200 || response.type !== 'basic') {
              return response;
            }
            const responseToCache = response.clone();
            caches.open(CACHE_NAME)
              .then((cache) => {
                cache.put(event.request, responseToCache);
              });
            return response;
          });
      })
  );
});
