// Self-hosted Video Background with blur overlay
let backgroundVideoConfig = {
  videoPath: null,      // Path to video file
  startTime: 0,         // Default start time
  overlayOpacity: 0.7,  // Default overlay opacity
  blurAmount: 8         // Default blur amount in pixels
};

let videoPlayer = null;
let initializationAttempts = 0;
const MAX_INIT_ATTEMPTS = 3;

// Load configuration from server - direct without cache
async function loadBackgroundVideoConfig() {
  try {
    console.log('Loading background video configuration...');
    
    // Add unique cache-busting parameter to prevent any cached results
    const cacheBuster = `_cb=${Date.now()}&_r=${Math.random().toString(36).substring(2, 15)}`;
    const response = await fetch(`/api/site-config?${cacheBuster}`, {
      headers: {
        'Cache-Control': 'no-cache, no-store, must-revalidate',
        'Pragma': 'no-cache'
      },
      cache: 'no-store'
    });
    
    if (response.ok) {
      const config = await response.json();
      if (config.backgroundVideo) {
        backgroundVideoConfig = {
          videoPath: config.backgroundVideo.videoPath || backgroundVideoConfig.videoPath,
          startTime: config.backgroundVideo.startTime ?? backgroundVideoConfig.startTime,
          overlayOpacity: config.backgroundVideo.overlayOpacity ?? backgroundVideoConfig.overlayOpacity,
          blurAmount: config.backgroundVideo.blurAmount ?? backgroundVideoConfig.blurAmount
        };
        
        console.log('Loaded background video config from server:', backgroundVideoConfig);
        
        // Update overlay styles
        updateOverlayStyles();
      }
    }
  } catch (error) {
    console.error('Error loading background video config:', error);
  }
  
  // Initialize the video player
  initializeVideoPlayer();
}

// Function to update overlay styles based on config
function updateOverlayStyles() {
  const overlay = document.querySelector('.video-overlay');
  if (overlay) {
    overlay.style.backgroundColor = `rgba(0, 0, 0, ${backgroundVideoConfig.overlayOpacity})`;
    overlay.style.backdropFilter = `blur(${backgroundVideoConfig.blurAmount}px)`;
    overlay.style.webkitBackdropFilter = `blur(${backgroundVideoConfig.blurAmount}px)`;
  }
}

// Initialize the video player
function initializeVideoPlayer() {
  if (!backgroundVideoConfig.videoPath) {
    return;
  }

  const videoContainer = document.getElementById('video-background');
  if (!videoContainer) {
    return;
  }
  
  try {
    // Create video element if it doesn't exist
    let videoElement = document.getElementById('video-background-player');
    if (!videoElement || !(videoElement instanceof HTMLVideoElement)) {
      // If the element exists but is not a video, remove it
      if (videoElement) {
        videoElement.remove();
      }
      
      // Create new video element
      videoElement = document.createElement('video');
      videoElement.id = 'video-background-player';
      videoElement.autoplay = true;
      videoElement.loop = true;
      videoElement.muted = true;
      videoElement.playsInline = true;
      videoElement.setAttribute('playsinline', '');
      videoElement.setAttribute('disablePictureInPicture', '');
      videoElement.setAttribute('webkit-playsinline', '');
      
      // Append to container
      videoContainer.prepend(videoElement);
    }
    
    // Set the source with cache busting
    const cacheBuster = Date.now();
    videoElement.src = `${backgroundVideoConfig.videoPath}?v=${cacheBuster}`;
    
    // Set the current time if specified
    if (backgroundVideoConfig.startTime > 0) {
      videoElement.currentTime = backgroundVideoConfig.startTime;
    }
    
    // Store reference to the video player
    videoPlayer = videoElement;
    
    // Set up event handlers
    setupVideoEventHandlers(videoElement);
    
    // Make sure video is visible
    videoContainer.style.visibility = 'visible';
    videoContainer.style.opacity = '1';
    
    // Make sure player is properly sized
    resizeVideoBackground();
    
    // Add a second resize attempt after layout stabilizes
    setTimeout(resizeVideoBackground, 500);
  } catch (e) {
    console.error('Error initializing video player:', e);
    
    // Try recovery if needed
    if (initializationAttempts < MAX_INIT_ATTEMPTS) {
      // Recreate video container
      const videoBackground = document.getElementById('video-background');
      if (videoBackground) {
        videoBackground.innerHTML = `
          <video id="video-background-player" autoplay loop muted playsinline></video>
          <div class="video-overlay"></div>
        `;
        
        // Update overlay styles
        updateOverlayStyles();
        
        // Try again after a delay
        setTimeout(() => {
          initializeVideoPlayer();
        }, 2000);
        
        initializationAttempts++;
      }
    }
  }
}

// Set up video event handlers
function setupVideoEventHandlers(videoElement) {
  // Play the video as soon as it's loaded
  videoElement.addEventListener('loadedmetadata', function() {
    videoElement.play().catch(error => {
      // Try with user interaction
      document.addEventListener('click', function playVideoOnce() {
        videoElement.play();
        document.removeEventListener('click', playVideoOnce);
      }, { once: true });
    });
    
    // Adjust the video sizing
    resizeVideoBackground();
  });
  
  // Monitor if video stops playing
  videoElement.addEventListener('pause', function() {
    // Try to resume if it paused unexpectedly
    if (!document.hidden) {
      videoElement.play();
    }
  });
}

// Additional mobile detection for better resize handling 
function isMobileDevice() {
  return /iPhone|iPad|iPod|Android|webOS|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent) 
      || window.innerWidth < 768;
}

// Improved resize function to ensure full coverage
function resizeVideoBackground() {
  const videoContainer = document.getElementById('video-background');
  if (!videoContainer) return;
  
  const videoElement = document.getElementById('video-background-player');
  if (!videoElement) return;
  
  const windowWidth = window.innerWidth;
  const windowHeight = window.innerHeight;
  const isMobile = isMobileDevice();
  
  // Ensure the container covers the window with some extra margin
  const extraMargin = isMobile ? 200 : 150;
  videoContainer.style.width = (windowWidth + extraMargin) + 'px';
  videoContainer.style.height = (windowHeight + extraMargin) + 'px';
  
  // Set video element to cover the entire container
  videoElement.style.width = '100%';
  videoElement.style.height = '100%';
  videoElement.style.objectFit = 'cover';
  videoElement.style.objectPosition = 'center center';
  
  // Add extra scaling for better coverage
  const baseScale = isMobile ? 1.3 : 1.2;
  videoElement.style.transform = `translate(-50%, -50%) scale(${baseScale})`;
}

// Check visibility and reinitialize if needed
function checkVisibilityAndInit() {
  if (document.visibilityState === 'visible') {
    // If video exists, make sure it's playing
    const videoElement = document.getElementById('video-background-player');
    if (videoElement && videoElement instanceof HTMLVideoElement) {
      if (videoElement.paused) {
        videoElement.play();
      }
    } 
    // Otherwise try to initialize
    else if (initializationAttempts < MAX_INIT_ATTEMPTS) {
      initializeVideoPlayer();
      initializationAttempts++;
    }
  }
}

// Initialize the video background
function initVideoBackground() {
  // Initial setup
  loadBackgroundVideoConfig();
  resizeVideoBackground();
  
  // Event handlers
  window.addEventListener('resize', resizeVideoBackground);
  window.addEventListener('orientationchange', function() {
    // Special handling for orientation changes on mobile
    setTimeout(resizeVideoBackground, 100);
    // Second attempt after layout stabilizes
    setTimeout(resizeVideoBackground, 500);
  });
  document.addEventListener('visibilitychange', checkVisibilityAndInit);
}

// Start once DOM is loaded
document.addEventListener('DOMContentLoaded', initVideoBackground);

// Additional initialization when window loads (backup)
window.addEventListener('load', () => {
  // If video not playing after page load, try initialization again
  const videoElement = document.getElementById('video-background-player');
  if ((!videoElement || videoElement.paused) && initializationAttempts < MAX_INIT_ATTEMPTS) {
    initializeVideoPlayer();
  }
});
