document.addEventListener('DOMContentLoaded', () => {
  // Load site configuration once at start
  loadSiteConfig();
  
  // Track video playback state
  window.videoIsPlaying = false;
  window.videoPlayerReady = false;
  window.lastVideoInteraction = 0;
  window.currentVideoId = null;
  
  // Add staggered animation to link items without waiting for overlay
  const links = document.querySelectorAll('.link-item');
  
  links.forEach((link, index) => {
    link.style.opacity = '0';
    link.style.transform = 'translateY(20px)';
    
    setTimeout(() => {
      link.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
      link.style.opacity = '1';
      link.style.transform = 'translateY(0)';
    }, 300 + (index * 120)); // Start sooner since no intro animation
  });
  
  // Add animation for video container
  const videoContainer = document.querySelector('.video-container');
  if (videoContainer) {
    videoContainer.style.opacity = '0';
    videoContainer.style.transform = 'translateY(30px)';
    
    setTimeout(() => {
      videoContainer.style.transition = 'opacity 0.8s ease, transform 0.8s ease';
      videoContainer.style.opacity = '1';
      videoContainer.style.transform = 'translateY(0)';
    }, 1000); // Start sooner since no intro animation
  }
  
  // Add animation for social icons
  const socialIcons = document.querySelectorAll('.social-icon');
  
  socialIcons.forEach((icon, index) => {
    icon.style.opacity = '0';
    icon.style.transform = 'translateY(15px)';
    
    setTimeout(() => {
      icon.style.transition = 'opacity 0.4s ease, transform 0.4s ease, color 0.3s ease, box-shadow 0.3s ease';
      icon.style.opacity = '1';
      icon.style.transform = 'translateY(0)';
    }, 200 + (index * 100));
  });
  
  // Update copyright year to current year
  document.getElementById('current-year').textContent = new Date().getFullYear();
  
  // Add toggle functionality for artist bio
  const infoToggle = document.getElementById('infoToggle');
  const artistBio = document.getElementById('artistBio');
  
  if (infoToggle && artistBio) {
    infoToggle.addEventListener('click', () => {
      infoToggle.classList.toggle('active');
      artistBio.classList.toggle('active');
    });
  }
  
  // Setup user interaction tracking for the video container
  setupVideoInteractionTracking();
  
  // Set up YouTube API for the featured video
  setupYouTubeAPI();
  
  // Set up BroadcastChannel for receiving admin updates
  if (typeof BroadcastChannel !== 'undefined') {
    const bc = new BroadcastChannel('cameo_content_updates');
    bc.onmessage = (event) => {
      if (event.data && event.data.type === 'content-update') {
        console.log('Content update notification received via BroadcastChannel');
        loadSiteConfig(true); // Force reload when admin makes changes
      }
    };
  }
  
  // Listen for localStorage changes (alternative to BroadcastChannel)
  window.addEventListener('storage', (event) => {
    if (event.key === 'adminChangesAt') {
      console.log('Admin changes detected via localStorage event');
      loadSiteConfig(true);
    }
  });
  
  // Force reload from server when visibility changes (user tabs back to page)
  document.addEventListener('visibilitychange', () => {
    if (document.visibilityState === 'visible') {
      console.log('Page is now visible, checking for updates...');
      loadSiteConfig(true);
    }
  });
});

// Setup tracking for user interactions with the video
function setupVideoInteractionTracking() {
  const videoContainer = document.querySelector('.youtube-embed');
  if (!videoContainer) return;
  
  // Track when user interacts with the video container
  videoContainer.addEventListener('mousedown', recordVideoInteraction);
  videoContainer.addEventListener('touchstart', recordVideoInteraction);
  
  // Also track focus and pointerdown events for better accessibility
  videoContainer.addEventListener('focus', recordVideoInteraction, true);
  videoContainer.addEventListener('pointerdown', recordVideoInteraction);
  
  // Track current video ID from iframe
  const iframe = videoContainer.querySelector('iframe');
  if (iframe) {
    const src = iframe.getAttribute('src') || '';
    const match = src.match(/embed\/([^?&]+)/);
    if (match && match[1]) {
      window.currentVideoId = match[1];
      console.log('Initial video ID:', window.currentVideoId);
    }
  }
}

// Record that user interacted with the video
function recordVideoInteraction() {
  window.lastVideoInteraction = Date.now();
  console.log('Video interaction recorded at', new Date().toISOString());
}

// Setup YouTube API for detecting video play state
function setupYouTubeAPI() {
  // Add YouTube iframe API if not already loaded
  if (!window.YT) {
    const tag = document.createElement('script');
    tag.src = 'https://www.youtube.com/iframe_api';
    const firstScriptTag = document.getElementsByTagName('script')[0];
    firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
  }
  
  // This will be called when the YT API is ready
  window.onYouTubeIframeAPIReady = function() {
    connectExistingYouTubePlayer();
  };
  
  // Listen for iframe changes to reconnect API
  const videoContainer = document.querySelector('.youtube-embed');
  if (videoContainer) {
    const observer = new MutationObserver(function(mutations) {
      mutations.forEach(function(mutation) {
        if (mutation.type === 'childList' && mutation.addedNodes.length) {
          // When iframe is added or changed
          setTimeout(connectExistingYouTubePlayer, 100);
        }
      });
    });
    
    observer.observe(videoContainer, { childList: true });
  }
}

// Connect to existing YouTube iframe
function connectExistingYouTubePlayer() {
  const iframe = document.querySelector('.youtube-embed iframe');
  if (!iframe) return;
  
  try {
    // Extract the video ID from the iframe src
    const src = iframe.getAttribute('src');
    const match = src.match(/embed\/([^?&]+)/);
    if (!match || !match[1]) return;
    
    const videoId = match[1];
    window.currentVideoId = videoId;
    
    // If YT API is ready, create a player instance
    if (window.YT && window.YT.Player) {
      // Create a player that connects to the existing iframe
      if (window.featuredVideoPlayer) {
        try {
          delete window.featuredVideoPlayer;
        } catch (e) {
          console.log('Error cleaning up old player:', e);
        }
      }
      
      window.featuredVideoPlayer = new YT.Player(iframe, {
        events: {
          'onStateChange': onPlayerStateChange,
          'onReady': onPlayerReady,
          'onError': function(event) {
            console.error('YouTube player error:', event.data);
            // Try to recover from YouTube errors
            if (event.data === 150 || event.data === 101) {
              console.log('Attempting to recover from embedding error...');
              // The video cannot be played in an embedded player
              // Try reloading the iframe with different parameters
              const container = document.querySelector('.youtube-embed');
              if (container) {
                const simpleIframe = document.createElement('iframe');
                simpleIframe.src = `https://www.youtube.com/embed/${videoId}?playsinline=1`;
                simpleIframe.setAttribute('allowfullscreen', 'allowfullscreen');
                simpleIframe.setAttribute('playsinline', '1');
                container.innerHTML = '';
                container.appendChild(simpleIframe);
              }
            }
          }
        }
      });
    }
  } catch (e) {
    console.error('Error connecting to YouTube iframe:', e);
    
    // Fallback for mobile browsers
    if (/iPhone|iPad|iPod|Android/i.test(navigator.userAgent)) {
      const container = document.querySelector('.youtube-embed');
      const videoId = window.currentVideoId;
      if (container && videoId) {
        console.log('Using simplified iframe for mobile device');
        const simpleIframe = document.createElement('iframe');
        simpleIframe.src = `https://www.youtube.com/embed/${videoId}?playsinline=1`;
        simpleIframe.setAttribute('allowfullscreen', 'allowfullscreen');
        simpleIframe.setAttribute('playsinline', '1');
        container.innerHTML = '';
        container.appendChild(simpleIframe);
      }
    }
  }
}

// When the player is ready
function onPlayerReady(event) {
  window.videoPlayerReady = true;
  console.log('YouTube player ready');
  
  // Check if the player is already playing (in case we reconnected to an active player)
  try {
    const state = event.target.getPlayerState();
    if (state === YT.PlayerState.PLAYING) {
      window.videoIsPlaying = true;
      console.log('Detected video is already playing');
    }
  } catch (e) {
    console.error('Error checking player state:', e);
  }
}

// Track video play state
function onPlayerStateChange(event) {
  // Update the global state based on player state
  // YT.PlayerState: PLAYING=1, PAUSED=2, ENDED=0, BUFFERING=3
  if (event.data === 1) { // Playing
    window.videoIsPlaying = true;
    // Record interaction when playback starts
    recordVideoInteraction();
    console.log('Featured video is now playing');
  } else if (event.data === 2) { // Paused
    window.videoIsPlaying = false;
    // Still record interaction when paused - user is engaging with the video
    recordVideoInteraction();
    console.log('Featured video is now paused');
  } else if (event.data === 0) { // Ended
    window.videoIsPlaying = false;
    console.log('Featured video has ended');
  } else if (event.data === 3) { // Buffering
    // Record interaction when buffering - user is likely waiting for video
    recordVideoInteraction();
    console.log('Featured video is buffering');
  }
}

// Load site configuration from server
async function loadSiteConfig(forceReload = false) {
  try {
    // Create a unique cache-busting parameter each time
    const timestamp = Date.now();
    const random = Math.random().toString(36).substring(2, 10);
    const cacheBuster = `_t=${timestamp}&_r=${random}`;
    
    const response = await fetch(`/api/site-config?${cacheBuster}`, {
      headers: {
        'Pragma': 'no-cache',
        'Cache-Control': 'no-cache, no-store, must-revalidate',
        'X-Requested-With': 'XMLHttpRequest'
      },
      // Set to 'no-store' to always go to network
      cache: 'no-store'
    });
    
    if (response.ok) {
      const config = await response.json();
      
      // Update the page content
      updatePageContent(config);
      
      // Set initial video ID from config
      if (config.video && config.video.videoId) {
        window.currentVideoId = config.video.videoId;
        console.log('Initial video ID from config:', window.currentVideoId);
      }
      
      // Return true to indicate successful load
      return true;
    } else {
      console.error('Failed to load site configuration');
      return false;
    }
  } catch (error) {
    console.error('Error loading site configuration:', error);
    return false;
  }
}

// Update page content based on configuration
function updatePageContent(config) {
  // Update profile
  if (config.profile) {
    document.querySelector('h1').textContent = config.profile.name;
    document.querySelector('.profile p').textContent = config.profile.subtitle;
    
    // Update profile image if it exists with cache busting
    if (config.profile.imagePath) {
      const profileImage = document.querySelector('.profile-image img');
      const timestamp = new Date().getTime();
      const imageSrc = config.profile.imagePath + 
        (config.profile.imagePath.includes('?') ? '&' : '?') + 
        `v=${timestamp}`;
      
      profileImage.src = imageSrc;
    } else {
      // Use SVG as default if no custom image is set
      const profileImage = document.querySelector('.profile-image img');
      profileImage.src = '/assets/Round.svg';
    }
  }
  
  // Update countdown section if it exists
  updateCountdownSection(config);
  
  // Update links
  if (config.links) {
    const linkMap = {
      'spotify': document.querySelector('.link-item.spotify'),
      'apple': document.querySelector('.link-item.apple'),
      'youtube': document.querySelector('.link-item.youtube'),
      'youtubeMusic': document.querySelector('.link-item.youtube-music'),
      'amazon': document.querySelector('.link-item.amazon')
    };
    
    for (const [key, value] of Object.entries(config.links)) {
      if (linkMap[key] && value) {
        linkMap[key].href = value;
      }
    }
  }
  
  // Update video - with improved protection against interruptions
  if (config.video) {
    // Update video title
    const videoHeading = document.querySelector('.video-container h2');
    if (videoHeading && config.video.title) {
      videoHeading.textContent = config.video.title;
    }
    
    const videoContainer = document.querySelector('.youtube-embed');
    if (videoContainer && config.video.videoId) {
      // 1. Check if the video is currently playing
      const isPlaying = window.videoIsPlaying === true;
      
      // 2. Check if the user has interacted with the video recently (within last 3 minutes)
      const interactionCooldownTime = 3 * 60 * 1000; // 3 minutes
      const recentInteraction = window.lastVideoInteraction > 0 && 
                               (Date.now() - window.lastVideoInteraction) < interactionCooldownTime;
      
      // 3. Check if the video ID has actually changed
      const videoIdChanged = window.currentVideoId !== config.video.videoId;
      
      console.log('Video update check:', { 
        isPlaying,
        recentInteraction, 
        timeSinceInteraction: window.lastVideoInteraction ? (Date.now() - window.lastVideoInteraction) / 1000 + 's' : 'never',
        currentId: window.currentVideoId,
        newId: config.video.videoId,
        videoIdChanged
      });
      
      // Only update if:
      // - Video is not playing AND
      // - No recent interaction AND
      // - The video ID has actually changed OR it's a forced initialization (when currentVideoId is null)
      if ((!isPlaying && !recentInteraction && videoIdChanged) || window.currentVideoId === null) {
        console.log('Updating featured video iframe');
        
        const timestamp = Date.now();
        // Enhanced parameters for better mobile compatibility and prevent caching
        const newSrc = `https://www.youtube.com/embed/${config.video.videoId}?playsinline=1&enablejsapi=1&origin=${encodeURIComponent(window.location.origin)}&_t=${timestamp}&_r=${Math.random().toString(36).substring(2,10)}&rel=0&modestbranding=1&widget_referrer=${encodeURIComponent(window.location.origin)}&hl=en`;
        
        // Create a new iframe to ensure clean reload with proper mobile attributes
        const newIframe = document.createElement('iframe');
        newIframe.src = newSrc;
        newIframe.title = config.video.title || 'Cameo - Latest Music Video';
        newIframe.frameBorder = '0';
        newIframe.allow = 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture';
        newIframe.allowFullscreen = true;
        newIframe.setAttribute('playsinline', '1'); // Important attribute for iOS
        newIframe.setAttribute('webkit-playsinline', '1'); // For older iOS versions
        
        // Replace the entire container contents with the new iframe
        videoContainer.innerHTML = '';
        videoContainer.appendChild(newIframe);
        
        // Update the current video ID
        window.currentVideoId = config.video.videoId;
        
        // Connect the new iframe to the YouTube API
        if (window.YT && window.YT.Player) {
          setTimeout(() => connectExistingYouTubePlayer(), 500);
        }
      } else {
        if (isPlaying) {
          console.log('Video is currently playing - skipping video iframe update');
        } else if (recentInteraction) {
          console.log('Recent video interaction detected - skipping video iframe update');
        } else if (!videoIdChanged) {
          console.log('Video ID unchanged - skipping video iframe update');
        }
      }
    }
  }
  
  // Update social links
  if (config.social) {
    const socialIcons = document.querySelectorAll('.social-icon');
    
    if (socialIcons.length >= 3) {
      if (config.social.instagram) {
        socialIcons[0].href = config.social.instagram;
      }
      
      if (config.social.facebook) {
        socialIcons[1].href = config.social.facebook;
      }
      
      if (config.social.tiktok) {
        socialIcons[2].href = config.social.tiktok;
      }
    }
  }
  
  // Update artist bio
  if (config.bio) {
    const bioParagraph = document.querySelector('.artist-bio p');
    if (bioParagraph && bioParagraph.textContent !== config.bio) {
      bioParagraph.textContent = config.bio;
    }
  }
  
  // Apply Google Analytics code
  if (config.seo && config.seo.gaTrackingCode) {
    // Direct insertion without DOMPurify
    insertAnalyticsCode(config.seo.gaTrackingCode);
  }
}

// Function to update the countdown section
function updateCountdownSection(config) {
  const countdownContainer = document.getElementById('countdown-container');
  if (!countdownContainer) return;
  
  console.log('Countdown config:', config.countdown);
  
  // Check if section should be shown at all
  if (config.countdown && config.countdown.enabled === true) {
    // Set the countdown title
    const countdownTitle = document.getElementById('countdown-title');
    if (countdownTitle) {
      countdownTitle.textContent = config.countdown.title || 'Upcoming Release';
    }
    
    // Set the release title
    const releaseTitle = document.getElementById('release-title');
    if (releaseTitle) {
      releaseTitle.textContent = config.countdown.releaseTitle || '';
    }
    
    // Set pre-save button text
    const presaveText = document.getElementById('presave-text');
    if (presaveText) {
      presaveText.textContent = config.countdown.presaveText || 'Pre-save on Spotify';
    }
    
    // Handle countdown image if available
    const countdownImageContainer = document.getElementById('countdown-image-container');
    const countdownImage = document.getElementById('countdown-image');
    
    if (countdownImageContainer && countdownImage) {
      if (config.countdown.imagePath) {
        countdownImage.src = config.countdown.imagePath;
        countdownImageContainer.style.display = 'block';
      } else {
        countdownImageContainer.style.display = 'none';
      }
    }
    
    // Check if timer should be shown
    const countdownTimerEl = document.querySelector('.countdown-timer');
    if (countdownTimerEl) {
      // Check both the showTimer and hideTimerNumbers flags
      if (config.countdown.showTimer && !config.countdown.hideTimerNumbers && config.countdown.releaseDate) {
        // Set up countdown timer
        setupCountdownTimer(config.countdown.releaseDate);
        countdownTimerEl.style.display = 'flex';
        console.log('Showing countdown timer');
      } else if (config.countdown.hideTimerNumbers) {
        // Hide the timer but keep the section
        countdownTimerEl.style.display = 'none';
        console.log('Hiding countdown timer - Timer numbers hidden by user choice');
        
        // Clean up any existing interval
        if (window.countdownInterval) {
          clearInterval(window.countdownInterval);
        }
      } else {
        // Hide the timer but keep the section
        countdownTimerEl.style.display = 'none';
        console.log('Hiding countdown timer - Timer disabled');
        
        // Clean up any existing interval
        if (window.countdownInterval) {
          clearInterval(window.countdownInterval);
        }
      }
    }
    
    // Set up Spotify pre-save button
    setupSpotifyPresave(config.countdown.upc);
    
    // Show the countdown section
    countdownContainer.style.display = 'block';
    console.log('Showing countdown section');
  } else {
    // Hide the countdown section
    countdownContainer.style.display = 'none';
    console.log('Hiding countdown section - Section disabled');
    
    // Clean up any existing countdown interval
    if (window.countdownInterval) {
      clearInterval(window.countdownInterval);
    }
  }
}

// Function to set up the countdown timer
function setupCountdownTimer(releaseDateStr) {
  const releaseDate = new Date(releaseDateStr);
  const countdownDays = document.getElementById('countdown-days');
  const countdownHours = document.getElementById('countdown-hours');
  const countdownMinutes = document.getElementById('countdown-minutes');
  const countdownSeconds = document.getElementById('countdown-seconds');
  
  // Update countdown every second
  const countdownInterval = setInterval(() => {
    const now = new Date();
    const timeDiff = releaseDate - now;
    
    if (timeDiff <= 0) {
      // Release time reached, stop the countdown
      clearInterval(countdownInterval);
      // Hide the countdown or show a "Now Available" message
      const countdownContainer = document.getElementById('countdown-container');
      if (countdownContainer) {
        // Option 1: Hide the countdown
        // countdownContainer.style.display = 'none';
        
        // Option 2: Show a "Now Available" message
        countdownContainer.innerHTML = `
          <h2 class="countdown-title">Now Available!</h2>
          <p class="release-title">${document.getElementById('release-title').textContent}</p>
          <button id="listen-button" class="presave-button">
            <i class="fab fa-spotify"></i> Listen on Spotify
          </button>
        `;
        
        // Add event listener to the new button
        const listenButton = document.getElementById('listen-button');
        if (listenButton) {
          listenButton.addEventListener('click', () => {
            const spotifyLink = document.querySelector('.link-item.spotify').href;
            if (spotifyLink) {
              window.open(spotifyLink, '_blank');
            }
          });
        }
      }
      return;
    }
    
    // Calculate days, hours, minutes, seconds
    const days = Math.floor(timeDiff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((timeDiff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    const minutes = Math.floor((timeDiff % (1000 * 60 * 60)) / (1000 * 60));
    const seconds = Math.floor((timeDiff % (1000 * 60)) / 1000);
    
    // Update the countdown display
    countdownDays.textContent = days.toString().padStart(2, '0');
    countdownHours.textContent = hours.toString().padStart(2, '0');
    countdownMinutes.textContent = minutes.toString().padStart(2, '0');
    countdownSeconds.textContent = seconds.toString().padStart(2, '0');
  }, 1000);
  
  // Store the interval ID to clear it if needed
  window.countdownInterval = countdownInterval;
}

// Function to set up Spotify pre-save
function setupSpotifyPresave(upc) {
  const presaveButton = document.getElementById('presave-button');
  if (!presaveButton) return;
  
  // Clear any existing event listeners
  presaveButton.replaceWith(presaveButton.cloneNode(true));
  
  // Get the new button reference
  const newPresaveButton = document.getElementById('presave-button');
  
  newPresaveButton.addEventListener('click', () => {
    if (!upc) {
      console.error('No UPC code provided for Spotify pre-save');
      return;
    }
    
    // Spotify pre-save URL format
    // For simplicity, we'll use a direct link to a pre-save service
    // In a real implementation, you would integrate with Spotify API
    const spotifyClientId = '8e3baa5667cd4a999a48ca060a7e7f58';
    
    // Create a popup for Spotify authorization
    const width = 450;
    const height = 730;
    const left = (window.screen.width / 2) - (width / 2);
    const top = (window.screen.height / 2) - (height / 2);
    
    const authUrl = `https://accounts.spotify.com/authorize?client_id=${spotifyClientId}&response_type=code&redirect_uri=${encodeURIComponent(window.location.origin + '/spotify-callback')}&scope=user-library-modify%20user-follow-modify&state=${upc}`;
    
    window.open(authUrl, 'Spotify Pre-save', `width=${width},height=${height},top=${top},left=${left}`);
  });
}

function insertAnalyticsCode(trackingCode) {
  if (!trackingCode || !trackingCode.trim()) return;
  
  try {
    console.log('Inserting Google Analytics code');
    
    // First, remove any existing analytics containers or scripts
    const existingContainers = document.querySelectorAll('#analytics-container');
    existingContainers.forEach(container => container.remove());
    
    const existingScript = document.getElementById('google-analytics');
    if (existingScript) {
      existingScript.remove();
    }
    
    // Check if code contains script tags
    if (trackingCode.includes('<script')) {
      // Create temporary element to parse HTML
      const tempDiv = document.createElement('div');
      tempDiv.innerHTML = trackingCode;
      
      // Extract all script elements
      const scripts = tempDiv.getElementsByTagName('script');
      const scriptElements = Array.from(scripts);
      
      // Process each script
      scriptElements.forEach(originalScript => {
        // Create a new script element
        const newScript = document.createElement('script');
        
        // Copy all attributes
        Array.from(originalScript.attributes).forEach(attr => {
          newScript.setAttribute(attr.name, attr.value);
        });
        
        // Copy inline content if any
        if (originalScript.innerHTML) {
          newScript.innerHTML = originalScript.innerHTML;
        }
        
        // Append to document
        document.body.appendChild(newScript);
      });
      
      console.log('Google Analytics scripts injected successfully');
    } else {
      // If it's just a tracking ID without script tags, create standard GA4 script
      const trackingId = trackingCode.trim();
      
      // Create gtag.js script
      const gtagScript = document.createElement('script');
      gtagScript.async = true;
      gtagScript.src = `https://www.googletagmanager.com/gtag/js?id=${trackingId}`;
      document.head.appendChild(gtagScript);
      
      // Create inline script
      const inlineScript = document.createElement('script');
      inlineScript.innerHTML = `
        window.dataLayer = window.dataLayer || [];
        function gtag(){dataLayer.push(arguments);}
        gtag('js', new Date());
        gtag('config', '${trackingId}');
      `;
      document.head.appendChild(inlineScript);
      
      console.log('Google Analytics initialized with ID:', trackingId);
    }
  } catch (error) {
    console.error('Error appending Google Analytics script:', error);
  }
}

// Function to automatically update the copyright year
function updateCopyrightYear() {
  const currentYear = new Date().getFullYear();
  const copyrightElements = document.querySelectorAll('footer p');
  
  copyrightElements.forEach(element => {
    if (element.textContent.includes('©')) {
      element.textContent = element.textContent.replace(/\d{4}/, currentYear);
    }
  });
}
