document.addEventListener('DOMContentLoaded', () => {
  // Check if logged in immediately
  if (!checkAuthenticationSync()) {
    console.log('Initial auth check failed, redirecting to login');
    return; // Stop execution if not authenticated
  }
  
  // Verify with the server and load data
  checkAuthenticationAsync();
  
  // Navigation between sections
  const navItems = document.querySelectorAll('.admin-nav li');
  const sections = document.querySelectorAll('.admin-section');
  
  navItems.forEach(item => {
    item.addEventListener('click', () => {
      const sectionId = item.getAttribute('data-section') + '-section';
      
      // Update active state on nav items
      navItems.forEach(nav => nav.classList.remove('active'));
      item.classList.add('active');
      
      // Show selected section, hide others
      sections.forEach(section => {
        section.classList.remove('active');
        if (section.id === sectionId) {
          section.classList.add('active');
        }
      });
      
      // Set the header title to match the section
      const sectionTitle = item.querySelector('span').textContent || item.textContent.trim();
      document.querySelector('.admin-header h1').textContent = sectionTitle;
    });
  });
  
  // Mobile optimization - collapse sections on small screens
  function optimizeForMobile() {
    if (window.innerWidth <= 768) {
      // Add collapsible behavior for section headings on mobile
      document.querySelectorAll('.admin-section h2').forEach(heading => {
        if (!heading.hasAttribute('data-mobile-initialized')) {
          const content = heading.nextElementSibling;
          
          // Add toggle indicator
          const indicator = document.createElement('i');
          indicator.className = 'fas fa-chevron-down';
          indicator.style.float = 'right';
          heading.appendChild(indicator);
          
          heading.style.cursor = 'pointer';
          heading.setAttribute('data-mobile-initialized', 'true');
          
          // Toggle content visibility when heading is clicked
          heading.addEventListener('click', () => {
            content.style.display = content.style.display === 'none' ? 'block' : 'none';
            indicator.className = content.style.display === 'none' ? 
              'fas fa-chevron-down' : 'fas fa-chevron-up';
          });
          
          // Initially keep content visible
          content.style.display = 'block';
        }
      });
    }
  }
  
  // Call on load and resize
  optimizeForMobile();
  window.addEventListener('resize', optimizeForMobile);
  
  // Logout button handler
  document.getElementById('logout-btn').addEventListener('click', () => {
    localStorage.removeItem('adminToken');
    window.location.href = '/login'; // Redirect to login page
  });
  
  // Save all changes button
  document.getElementById('save-all-btn').addEventListener('click', saveAllChanges);
  
  // Update profile image preview when file selected
  const profileImageInput = document.getElementById('profileImage');
  if (profileImageInput) {
    profileImageInput.addEventListener('change', (e) => {
      const file = e.target.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = (e) => {
          document.getElementById('profileImagePreview').src = e.target.result;
        };
        reader.readAsDataURL(file);
      }
    });
  }
  
  // Click handler for "Choose Image" button
  const uploadBtn = document.querySelector('.upload-btn');
  if (uploadBtn) {
    uploadBtn.addEventListener('click', () => {
      document.getElementById('profileImage').click();
    });
  }

  // Click handler for "Remove Image" button
  const removeBtn = document.getElementById('removeImageBtn');
  if (removeBtn) {
    removeBtn.addEventListener('click', removeProfileImage);
  }
  
  // Click handler for "Choose Video" button - improved to ensure it works
  const chooseVideoBtn = document.getElementById('chooseVideoBtn');
  if (chooseVideoBtn) {
    chooseVideoBtn.addEventListener('click', () => {
      const fileInput = document.getElementById('backgroundVideo');
      if (fileInput) {
        fileInput.click();
      }
    });
  }

  // Click handler for "Remove Video" button
  const removeVideoBtn = document.getElementById('removeVideoBtn');
  if (removeVideoBtn) {
    removeVideoBtn.addEventListener('click', removeBackgroundVideo);
  }
  
  // Background Video file change handler - enhanced with error handling
  const backgroundVideoInput = document.getElementById('backgroundVideo');
  if (backgroundVideoInput) {
    backgroundVideoInput.addEventListener('change', (e) => {
      const file = e.target.files[0];
      if (file) {
        console.log('Video file selected:', file.name, file.size, file.type);
        
        // Check file type
        if (!['video/mp4', 'video/webm', 'video/ogg'].includes(file.type)) {
          showStatusMessage('Please select a valid video file (MP4, WebM, or Ogg)', true);
          backgroundVideoInput.value = '';
          return;
        }
        
        // Check file size (50MB limit)
        const maxSize = 50 * 1024 * 1024; // 50MB in bytes
        if (file.size > maxSize) {
          showStatusMessage('Video file size must be under 50MB', true);
          backgroundVideoInput.value = '';
          return;
        }
        
        // Create URL and update preview
        const reader = new FileReader();
        reader.onload = (e) => {
          const videoPreview = document.getElementById('backgroundVideoPreview');
          videoPreview.src = e.target.result;
          videoPreview.style.display = 'block';
          
          // Hide thumbnail info since we have a video
          const thumbnailInfo = document.getElementById('videoThumbnailInfo');
          if (thumbnailInfo) {
            thumbnailInfo.style.display = 'none';
          }
          
          videoPreview.load();
          videoPreview.play().catch(err => console.warn('Could not autoplay preview', err));
        };
        reader.readAsDataURL(file);
      }
    });
  }

  // Add play/pause functionality for video preview
  const playVideoBtn = document.getElementById('playVideoBtn');
  if (playVideoBtn) {
    playVideoBtn.addEventListener('click', () => {
      const videoPreview = document.getElementById('backgroundVideoPreview');
      if (videoPreview && videoPreview.paused) {
        videoPreview.play().catch(err => console.warn('Could not play video:', err));
      }
    });
  }

  const pauseVideoBtn = document.getElementById('pauseVideoBtn');
  if (pauseVideoBtn) {
    pauseVideoBtn.addEventListener('click', () => {
      const videoPreview = document.getElementById('backgroundVideoPreview');
      if (videoPreview && !videoPreview.paused) {
        videoPreview.pause();
      }
    });
  }
  
  // Video ID preview handler
  const videoIdInput = document.getElementById('videoId');
  if (videoIdInput) {
    videoIdInput.addEventListener('input', updateVideoPreview);
  }
  
  // Background Video preview and range input handlers
  const backgroundVideoId = document.getElementById('backgroundVideoId');
  if (backgroundVideoId) {
    backgroundVideoId.addEventListener('input', updateBackgroundVideoPreview);
  }
  
  // Handle opacity range input
  const opacityRange = document.getElementById('backgroundVideoOpacity');
  if (opacityRange) {
    opacityRange.addEventListener('input', () => {
      document.getElementById('opacityValue').textContent = opacityRange.value;
      updateBackgroundOverlayPreview();
    });
  }
  
  // Handle blur range input
  const blurRange = document.getElementById('backgroundVideoBlur');
  if (blurRange) {
    blurRange.addEventListener('input', () => {
      document.getElementById('blurValue').textContent = blurRange.value;
      updateBackgroundOverlayPreview();
    });
  }
  
  // Update credentials button
  const updateCredentialsBtn = document.getElementById('updateCredentials');
  if (updateCredentialsBtn) {
    updateCredentialsBtn.addEventListener('click', updateLoginCredentials);
  }
  
  // Character counters for SEO fields
  const metaTitle = document.getElementById('metaTitle');
  const metaDescription = document.getElementById('metaDescription');
  
  if (metaTitle) {
    setupCharacterCounter(metaTitle, 60);
    metaTitle.addEventListener('input', () => updateCharacterCounter(metaTitle, 60));
  }
  
  if (metaDescription) {
    setupCharacterCounter(metaDescription, 160);
    metaDescription.addEventListener('input', () => updateCharacterCounter(metaDescription, 160));
  }

  // Initialize countdown toggle button
  const toggleCountdownBtn = document.getElementById('toggle-countdown-btn');
  if (toggleCountdownBtn) {
    toggleCountdownBtn.addEventListener('click', toggleCountdownStatus);
  }

  // Countdown image upload handlers
  const countdownImageInput = document.getElementById('countdownImage');
  if (countdownImageInput) {
    countdownImageInput.addEventListener('change', (e) => {
      const file = e.target.files[0];
      if (file) {
        const reader = new FileReader();
        reader.onload = (e) => {
          document.getElementById('countdownImagePreview').src = e.target.result;
        };
        reader.readAsDataURL(file);
      }
    });
  }

  // Click handler for "Choose Countdown Image" button
  const chooseCountdownImageBtn = document.getElementById('chooseCountdownImageBtn');
  if (chooseCountdownImageBtn) {
    chooseCountdownImageBtn.addEventListener('click', () => {
      document.getElementById('countdownImage').click();
    });
  }

  // Click handler for "Remove Countdown Image" button
  const removeCountdownImageBtn = document.getElementById('removeCountdownImageBtn');
  if (removeCountdownImageBtn) {
    removeCountdownImageBtn.addEventListener('click', removeCountdownImage);
  }
});

// Synchronous check for token existence
function checkAuthenticationSync() {
  // Check localStorage
  const localToken = localStorage.getItem('adminToken');
  
  // Check URL parameters (for first login)
  const urlParams = new URLSearchParams(window.location.search);
  const urlToken = urlParams.get('token');
  
  // Use any available token
  const token = localToken || urlToken;
  
  // If URL has token but localStorage doesn't, save it
  if (!localToken && urlToken) {
    console.log('Saving token from URL to localStorage');
    localStorage.setItem('adminToken', urlToken);
  }
  
  // If we have a token in URL, remove it from URL for security
  if (urlToken) {
    // Replace URL without reloading
    const newUrl = window.location.pathname;
    window.history.replaceState({}, document.title, newUrl);
  }
  
  if (!token) {
    // Check for redirect loop by looking for a counter in session storage
    const redirectCount = parseInt(sessionStorage.getItem('adminRedirectCount') || '0');
    if (redirectCount > 3) {
      console.error('Detected potential redirect loop, showing error instead of redirecting');
      document.body.innerHTML = '<div style="padding: 30px; text-align: center;"><h1>Authentication Error</h1>' +
        '<p>There was a problem with your authentication. Please try clearing your browser cache and cookies, then visit the ' +
        '<a href="/login">login page</a> directly.</p></div>';
      return true; // Return true to prevent further redirects
    }
    
    // Increment redirect counter
    sessionStorage.setItem('adminRedirectCount', (redirectCount + 1).toString());
    
    console.log('No authentication token found, redirecting to login');
    window.location.href = '/login';
    return false;
  }
  
  // If we have a token, reset the redirect counter
  sessionStorage.removeItem('adminRedirectCount');
  
  return true;
}

// Asynchronous check with the server
async function checkAuthenticationAsync() {
  const token = localStorage.getItem('adminToken');
  if (!token) {
    console.error('No token found in async check, redirecting');
    window.location.href = '/login';
    return false;
  }
  
  try {
    // Add cache busting parameter
    const cacheBuster = `_nocache=${Date.now()}_${Math.random().toString(36).substring(2)}`;
    
    console.log('Verifying token with server...');
    const response = await fetch(`/api/admin/verify?${cacheBuster}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Cache-Control': 'no-cache, no-store, must-revalidate',
        'Pragma': 'no-cache'
      },
      cache: 'no-store'
    });
    
    const data = await response.json();
    
    if (response.ok && data.valid) {
      // Token is valid, load site data
      console.log('Token verified, loading site data');
      loadSiteData();
      return true;
    } else {
      console.log('Token invalid or expired');
      // Clear token and redirect
      localStorage.removeItem('adminToken');
      
      // Check for redirect loop
      const redirectCount = parseInt(sessionStorage.getItem('adminRedirectCount') || '0');
      if (redirectCount > 3) {
        console.error('Detected potential redirect loop in async check');
        document.body.innerHTML = '<div style="padding: 30px; text-align: center;"><h1>Authentication Error</h1>' +
          '<p>Your session has expired or is invalid. Please try clearing your browser cache and cookies, then visit the ' +
          '<a href="/login">login page</a> directly.</p></div>';
        return false;
      }
      
      // Increment redirect counter
      sessionStorage.setItem('adminRedirectCount', (redirectCount + 1).toString());
      
      window.location.href = `/login?expired=${Date.now()}`;
      return false;
    }
  } catch (err) {
    console.error('Error verifying authentication:', err);
    
    // On network error, we'll continue with local token
    // This prevents being logged out due to temporary connectivity issues
    console.log('Continuing with local token due to network error');
    loadSiteData();
    return true;
  }
}

// Load all site data from config
async function loadSiteData() {
  try {
    const token = localStorage.getItem('adminToken');
    // Add unique cache-busting parameter
    const cacheBuster = `_t=${Date.now()}&_r=${Math.random().toString(36).substring(2, 15)}`;
    
    const response = await fetch(`/api/admin/site-data?${cacheBuster}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Pragma': 'no-cache',
        'Cache-Control': 'no-cache, no-store, must-revalidate',
        'X-Requested-With': 'XMLHttpRequest'
      },
      cache: 'no-store'
    });
    
    if (response.ok) {
      const data = await response.json();
      
      // Populate form fields with data
      // Profile
      document.getElementById('artistName').value = data.profile.name || '';
      document.getElementById('subtitle').value = data.profile.subtitle || '';
      if (data.profile.imagePath) {
        document.getElementById('profileImagePreview').src = data.profile.imagePath;
      } else {
        // Set default SVG image if no custom image is set
        document.getElementById('profileImagePreview').src = '/assets/Round.svg';
      }
      
      // Links
      document.getElementById('spotifyLink').value = data.links.spotify || '';
      document.getElementById('appleLink').value = data.links.apple || '';
      document.getElementById('youtubeLink').value = data.links.youtube || '';
      document.getElementById('ytMusicLink').value = data.links.youtubeMusic || '';
      document.getElementById('amazonLink').value = data.links.amazon || '';
      
      // Countdown
      if (data.countdown) {
        // Set the checkbox to control the timer display
        document.getElementById('enableCountdown').checked = data.countdown.showTimer || false;
        document.getElementById('countdownTitle').value = data.countdown.title || 'Upcoming Release';
        document.getElementById('releaseDate').value = data.countdown.releaseDate || '';
        document.getElementById('spotifyUPC').value = data.countdown.upc || '';
        document.getElementById('releaseTitle').value = data.countdown.releaseTitle || '';
        document.getElementById('presaveText').value = data.countdown.presaveText || 'Pre-save on Spotify';
        
        // Set hide timer numbers checkbox
        if (document.getElementById('hideTimerNumbers')) {
          document.getElementById('hideTimerNumbers').checked = data.countdown.hideTimerNumbers || false;
        }
        
        // Set countdown image if available
        if (data.countdown.imagePath) {
          document.getElementById('countdownImagePreview').src = data.countdown.imagePath;
        } else {
          document.getElementById('countdownImagePreview').src = '/assets/placeholder-image.png';
        }
        
        // Update toggle button based on section visibility, not timer display
        updateCountdownToggleButton(data.countdown.enabled || false);
      } else {
        updateCountdownToggleButton(false);
      }
      
      // Video
      document.getElementById('videoTitle').value = data.video.title || '';
      document.getElementById('videoId').value = data.video.videoId || '';
      updateVideoPreview();
      
      // Background Video
      if (data.backgroundVideo) {
        if (data.backgroundVideo.videoPath) {
          const videoPreview = document.getElementById('backgroundVideoPreview');
          const thumbnailInfo = document.getElementById('videoThumbnailInfo');
          
          if (videoPreview) {
            // Add timestamp to prevent caching
            videoPreview.src = `${data.backgroundVideo.videoPath}?t=${Date.now()}`;
            videoPreview.style.display = 'block';
            videoPreview.load();
            
            // Hide the thumbnail info since we have a video
            if (thumbnailInfo) {
              thumbnailInfo.style.display = 'none';
            }
            
            videoPreview.play().catch(e => console.warn('Could not autoplay video preview:', e));
          }
        } else {
          // No video path - show the thumbnail info
          const thumbnailInfo = document.getElementById('videoThumbnailInfo');
          if (thumbnailInfo) {
            thumbnailInfo.style.display = 'flex';
          }
        }
        
        if (data.backgroundVideo.startTime !== undefined) {
          document.getElementById('backgroundVideoStartTime').value = data.backgroundVideo.startTime;
        }
        
        if (data.backgroundVideo.overlayOpacity !== undefined) {
          const opacityElem = document.getElementById('backgroundVideoOpacity');
          opacityElem.value = data.backgroundVideo.overlayOpacity;
          document.getElementById('opacityValue').textContent = opacityElem.value;
        }
        
        if (data.backgroundVideo.blurAmount !== undefined) {
          const blurElem = document.getElementById('backgroundVideoBlur');
          blurElem.value = data.backgroundVideo.blurAmount;
          document.getElementById('blurValue').textContent = blurElem.value;
        }
        
        // Update preview overlay
        updateBackgroundOverlayPreview();
      }
      
      // Social
      document.getElementById('instagramLink').value = data.social.instagram || '';
      document.getElementById('facebookLink').value = data.social.facebook || '';
      document.getElementById('tiktokLink').value = data.social.tiktok || '';
      
      // Bio
      document.getElementById('artistBio').value = data.bio || '';
      
      // SEO & Analytics - Improved handling
      if (data.seo) {
        // Set website title if it exists
        if (document.getElementById('websiteTitle')) {
          document.getElementById('websiteTitle').value = data.seo.websiteTitle || '';
        }
        
        // Set form values if they exist
        if (document.getElementById('metaTitle')) {
          document.getElementById('metaTitle').value = data.seo.metaTitle || '';
          updateCharacterCounter(document.getElementById('metaTitle'), 60);
        }
        
        if (document.getElementById('metaDescription')) {
          document.getElementById('metaDescription').value = data.seo.metaDescription || '';
          updateCharacterCounter(document.getElementById('metaDescription'), 160);
        }
        
        if (document.getElementById('gaTrackingId')) {
          document.getElementById('gaTrackingId').value = data.seo.gaTrackingId || '';
        }
        
        if (document.getElementById('gaTrackingCode')) {
          document.getElementById('gaTrackingCode').value = data.seo.gaTrackingCode || '';
        }
      }
      
    } else {
      showStatusMessage('Failed to load site data', true);
    }
  } catch (err) {
    console.error('Error loading site data:', err);
    showStatusMessage('Connection error. Please refresh the page.', true);
  }
}

// Update the toggle button state
function updateCountdownToggleButton(isEnabled) {
  const toggleButton = document.getElementById('toggle-countdown-btn');
  const statusText = document.getElementById('countdown-toggle-status');
  
  if (!toggleButton || !statusText) return;
  
  if (isEnabled) {
    toggleButton.classList.add('enabled');
    statusText.textContent = 'Countdown: On';
  } else {
    toggleButton.classList.remove('enabled');
    statusText.textContent = 'Countdown: Off';
  }
}

// Toggle countdown status and save
async function toggleCountdownStatus() {
  const token = localStorage.getItem('adminToken');
  if (!token) {
    checkAuthenticationAsync();
    return;
  }
  
  try {
    // Get current site data
    const cacheBuster = `_t=${Date.now()}&_r=${Math.random().toString(36).substring(2, 15)}`;
    const response = await fetch(`/api/admin/site-data?${cacheBuster}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Cache-Control': 'no-cache, no-store, must-revalidate'
      },
      cache: 'no-store'
    });
    
    if (!response.ok) {
      showStatusMessage('Failed to retrieve current site data', true);
      return;
    }
    
    const siteData = await response.json();
    
    // Initialize countdown property if it doesn't exist
    if (!siteData.countdown) {
      siteData.countdown = {
        enabled: false,
        showTimer: false,
        title: 'Upcoming Release',
        releaseDate: '',
        upc: '',
        releaseTitle: '',
        presaveText: 'Pre-save on Spotify'
      };
    }
    
    // Toggle the enabled status (section visibility)
    siteData.countdown.enabled = !siteData.countdown.enabled;
    
    // Do NOT change the timer status (that's controlled by the checkbox)
    // siteData.countdown.showTimer stays the same
    
    // Update the toggle button appearance
    updateCountdownToggleButton(siteData.countdown.enabled);
    
    // Save the changes
    const saveResponse = await fetch('/api/admin/save-site-data', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
        'Cache-Control': 'no-cache, no-store, must-revalidate'
      },
      body: JSON.stringify(siteData)
    });
    
    if (saveResponse.ok) {
      showStatusMessage(`Countdown section ${siteData.countdown.enabled ? 'enabled' : 'disabled'}`);
      
      // Notify other windows about the change
      try {
        const timestamp = Date.now().toString();
        localStorage.setItem('adminChangesAt', timestamp);
        localStorage.setItem('adminChangesData', JSON.stringify({
          timestamp,
          type: 'countdown-toggle',
          enabled: siteData.countdown.enabled
        }));
        
        if (typeof BroadcastChannel !== 'undefined') {
          const bc = new BroadcastChannel('cameo_content_updates');
          bc.postMessage({
            type: 'content-update',
            timestamp: timestamp
          });
          setTimeout(() => bc.close(), 1000);
        }
      } catch (e) {
        console.warn('Failed to notify other tabs:', e);
      }
    } else {
      showStatusMessage('Failed to update countdown status', true);
    }
  } catch (err) {
    console.error('Error toggling countdown:', err);
    showStatusMessage('Error toggling countdown status', true);
  }
}

// Show status message function
function showStatusMessage(message, isError = false) {
  const statusMessage = document.getElementById('statusMessage');
  const statusText = document.getElementById('statusText');
  
  statusText.textContent = message;
  
  if (isError) {
    statusMessage.classList.add('error');
  } else {
    statusMessage.classList.remove('error');
  }
  
  statusMessage.classList.add('show');
  
  setTimeout(() => {
    statusMessage.classList.remove('show');
  }, 3000);
}

// Save all changes to the server
async function saveAllChanges() {
  const token = localStorage.getItem('adminToken');
  if (!token) {
    checkAuthenticationAsync();
    return;
  }
  
  // Show saving indicator
  showStatusMessage('Saving changes...');
  
  // Clean up tracking code - remove unnecessary line breaks and trim
  let gaTrackingCode = document.getElementById('gaTrackingCode').value;
  gaTrackingCode = gaTrackingCode.trim();
  
  // Basic validation for tracking code
  if (gaTrackingCode && !gaTrackingCode.includes('<script')) {
    showStatusMessage('Google Analytics code should include script tags', true);
    return;
  }
  
  try {
    // First, get the current site configuration to preserve data
    const cacheBuster = `_t=${Date.now()}&_r=${Math.random().toString(36).substring(2, 15)}`;
    const currentDataResponse = await fetch(`/api/admin/site-data?${cacheBuster}`, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Pragma': 'no-cache',
        'Cache-Control': 'no-cache, no-store, must-revalidate'
      },
      cache: 'no-store'
    });
    
    if (!currentDataResponse.ok) {
      showStatusMessage('Failed to retrieve current site data', true);
      return;
    }
    
    // Get existing data
    const existingSiteData = await currentDataResponse.json();
    
    // Gather new form data
    const newFormData = {
      profile: {
        name: document.getElementById('artistName').value,
        subtitle: document.getElementById('subtitle').value
      },
      links: {
        spotify: document.getElementById('spotifyLink').value,
        apple: document.getElementById('appleLink').value,
        youtube: document.getElementById('youtubeLink').value,
        youtubeMusic: document.getElementById('ytMusicLink').value,
        amazon: document.getElementById('amazonLink').value
      },
      // Add the countdown configuration with new fields
      countdown: {
        enabled: document.getElementById('enableCountdown').checked ? true : existingSiteData.countdown?.enabled || false,
        showTimer: document.getElementById('enableCountdown').checked,
        hideTimerNumbers: document.getElementById('hideTimerNumbers').checked,
        title: document.getElementById('countdownTitle').value,
        releaseDate: document.getElementById('releaseDate').value,
        upc: document.getElementById('spotifyUPC').value,
        releaseTitle: document.getElementById('releaseTitle').value,
        presaveText: document.getElementById('presaveText').value
      },
      video: {
        title: document.getElementById('videoTitle').value,
        videoId: document.getElementById('videoId').value,
        noCache: true // Always set video to no-cache
      },
      backgroundVideo: {
        startTime: parseInt(document.getElementById('backgroundVideoStartTime').value) || 0,
        overlayOpacity: parseFloat(document.getElementById('backgroundVideoOpacity').value),
        blurAmount: parseInt(document.getElementById('backgroundVideoBlur').value),
        noCache: true // Always set background video to no-cache
      },
      social: {
        instagram: document.getElementById('instagramLink').value,
        facebook: document.getElementById('facebookLink').value,
        tiktok: document.getElementById('tiktokLink').value,
        noCache: true // Always set social links to no-cache
      },
      bio: document.getElementById('artistBio').value,
      seo: {
        websiteTitle: document.getElementById('websiteTitle').value,
        metaTitle: document.getElementById('metaTitle').value,
        metaDescription: document.getElementById('metaDescription').value,
        gaTrackingId: document.getElementById('gaTrackingId').value,
        gaTrackingCode: gaTrackingCode
      },
      // Add no-cache setting for all content
      cacheSettings: {
        cacheBackgroundVideo: false, // Always disable caching for background video
        lastUpdated: new Date().toISOString()
      }
    };
    
    // Create merged data by combining existing data with form data
    const mergedSiteData = {
      // Start with all existing data
      ...existingSiteData,
      // Update profile section
      profile: {
        ...existingSiteData.profile,
        ...newFormData.profile
      },
      // Update links section
      links: {
        ...existingSiteData.links,
        ...newFormData.links
      },
      // Update countdown section
      countdown: {
        ...existingSiteData.countdown,
        ...newFormData.countdown
      },
      // Update video section
      video: {
        ...existingSiteData.video,
        ...newFormData.video
      },
      // Update backgroundVideo section, preserving videoPath if not changed
      backgroundVideo: {
        ...existingSiteData.backgroundVideo,
        ...newFormData.backgroundVideo
      },
      // Update social section
      social: {
        ...existingSiteData.social,
        ...newFormData.social
      },
      // Update bio
      bio: newFormData.bio,
      // Update SEO section
      seo: {
        ...existingSiteData.seo,
        ...newFormData.seo
      },
      // Update cache settings
      cacheSettings: newFormData.cacheSettings
    };
    
    // Handle profile image upload if there's a new image
    const profileImage = document.getElementById('profileImage').files[0];
    if (profileImage) {
      const formData = new FormData();
      formData.append('profileImage', profileImage);
      
      try {
        const uploadResponse = await fetch('/api/admin/upload-image', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`
          },
          body: formData
        });
        
        if (uploadResponse.ok) {
          const imageData = await uploadResponse.json();
          mergedSiteData.profile.imagePath = imageData.imagePath;
        } else {
          showStatusMessage('Failed to upload image', true);
        }
      } catch (err) {
        console.error('Image upload error:', err);
        showStatusMessage('Error uploading image', true);
      }
    } else {
      // If the preview image is the default, make sure we mark it as removed
      const previewSrc = document.getElementById('profileImagePreview').src;
      if (previewSrc.includes('Round.svg')) {
        mergedSiteData.profile.useDefaultImage = true;
      }
    }
    
    // Handle background video upload if there's a new video
    const backgroundVideo = document.getElementById('backgroundVideo').files[0];
    if (backgroundVideo) {
      const videoFormData = new FormData();
      videoFormData.append('backgroundVideo', backgroundVideo);
      
      try {
        const uploadResponse = await fetch('/api/admin/upload-video', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`
          },
          body: videoFormData
        });
        
        if (uploadResponse.ok) {
          const videoData = await uploadResponse.json();
          mergedSiteData.backgroundVideo.videoPath = videoData.videoPath;
        } else {
          showStatusMessage('Failed to upload video', true);
          return; // Stop saving if video upload fails
        }
      } catch (err) {
        console.error('Video upload error:', err);
        showStatusMessage('Error uploading video', true);
        return; // Stop saving if video upload fails
      }
    }
    
    // Handle countdown image upload if there's a new image
    const countdownImage = document.getElementById('countdownImage').files[0];
    if (countdownImage) {
      const formData = new FormData();
      formData.append('countdownImage', countdownImage);
      
      try {
        const uploadResponse = await fetch('/api/admin/upload-countdown-image', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`
          },
          body: formData
        });
        
        if (uploadResponse.ok) {
          const imageData = await uploadResponse.json();
          mergedSiteData.countdown.imagePath = imageData.imagePath;
        } else {
          showStatusMessage('Failed to upload countdown image', true);
        }
      } catch (err) {
        console.error('Countdown image upload error:', err);
        showStatusMessage('Error uploading countdown image', true);
      }
    }
    
    // If the preview image is the placeholder, check if we need to mark it as removed
    const previewSrc = document.getElementById('countdownImagePreview').src;
    if (previewSrc.includes('placeholder-image.png')) {
      // Only if we had a previous image
      if (existingSiteData.countdown && existingSiteData.countdown.imagePath) {
        mergedSiteData.countdown.imagePath = null;
      }
    }
    
    // Save all site data
    try {
      // Add unique identifier to prevent any caching
      const saveCacheBuster = `_t=${Date.now()}&_r=${Math.random().toString(36).substring(2, 15)}`;
      
      // Add cache control headers to prevent caching of this request
      const saveResponse = await fetch(`/api/admin/save-site-data?${saveCacheBuster}`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
          'Pragma': 'no-cache',
          'Cache-Control': 'no-cache, no-store, must-revalidate',
          'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify(mergedSiteData),
        cache: 'no-store'
      });
      
      if (saveResponse.ok) {
        const responseData = await saveResponse.json();
        showStatusMessage('Changes saved successfully!');
        
        // Try to update other browser windows/tabs
        try {
          // Use the server's generated cacheBuster for consistency
          const timestamp = responseData.cacheBuster || Date.now().toString();
          
          // Add an extra force reload flag
          localStorage.setItem('adminChangesAt', timestamp);
          localStorage.setItem('adminChangesData', JSON.stringify({
            timestamp,
            type: 'site-update',
            version: timestamp,
            forceReload: true
          }));
          
          // Force a reload of the current page to see changes immediately
          setTimeout(() => {
            window.location.reload(true);
          }, 1500);
          
          // Also try using the BroadcastChannel API for more reliable cross-tab communication
          if (typeof BroadcastChannel !== 'undefined') {
            const bc = new BroadcastChannel('cameo_content_updates');
            bc.postMessage({
              type: 'content-update',
              timestamp: timestamp
            });
            
            // Close the channel after sending
            setTimeout(() => bc.close(), 1000);
          }
        } catch (e) {
          // Ignore storage errors
          console.warn('Failed to notify other tabs:', e);
        }
      } else {
        const errorData = await saveResponse.json();
        showStatusMessage(errorData.message || 'Failed to save changes', true);
      }
    } catch (err) {
      console.error('Save error:', err);
      showStatusMessage('Connection error. Changes not saved.', true);
    }
  } catch (err) {
    console.error('Error preparing data for save:', err);
    showStatusMessage('Error preparing data for save.', true);
  }
}

// Update video preview based on entered YouTube ID
function updateVideoPreview() {
  const videoId = document.getElementById('videoId').value.trim();
  const videoPreview = document.getElementById('videoPreview');
  const videoPlaceholder = document.querySelector('.video-placeholder');
  
  if (videoId) {
    videoPreview.innerHTML = `<iframe src="https://www.youtube.com/embed/${videoId}" allowfullscreen></iframe>`;
    videoPreview.style.display = 'block';
    videoPlaceholder.style.display = 'none';
  } else {
    videoPreview.style.display = 'none';
    videoPlaceholder.style.display = 'flex';
  }
}

// Update background video preview
function updateBackgroundVideoPreview() {
  const videoId = document.getElementById('backgroundVideoId').value.trim();
  const videoPreview = document.getElementById('backgroundVideoPreview');
  const videoPlaceholder = document.querySelector('#backgroundVideoPreviewContainer .video-placeholder');
  
  if (videoId) {
    videoPreview.innerHTML = `<iframe src="https://www.youtube.com/embed/${videoId}" allowfullscreen></iframe>`;
    videoPreview.style.display = 'block';
    videoPlaceholder.style.display = 'none';
  } else {
    videoPreview.style.display = 'none';
    videoPlaceholder.style.display = 'flex';
  }
}

// Function to update overlay styles based on range inputs
function updateBackgroundOverlayPreview() {
  const opacity = document.getElementById('backgroundVideoOpacity').value;
  const blur = document.getElementById('backgroundVideoBlur').value;
  
  const videoPreview = document.getElementById('backgroundVideoPreview');
  if (videoPreview) {
    // Add a style element for preview overlay if it doesn't exist
    let previewStyle = document.getElementById('preview-overlay-style');
    if (!previewStyle) {
      previewStyle = document.createElement('style');
      previewStyle.id = 'preview-overlay-style';
      document.head.appendChild(previewStyle);
    }
    
    // Update the styles
    previewStyle.textContent = `
      #backgroundVideoPreview {
        filter: blur(${blur}px);
        opacity: ${1 - opacity};
      }
    `;
  }
}

// Update admin login credentials
async function updateLoginCredentials() {
  const username = document.getElementById('adminUsername').value;
  const password = document.getElementById('adminPassword').value;
  const confirmPassword = document.getElementById('confirmPassword').value;
  
  if (!username && !password) {
    showStatusMessage('No changes to save', true);
    return;
  }
  
  if (password && password !== confirmPassword) {
    showStatusMessage('Passwords do not match', true);
    return;
  }
  
  try {
    const token = localStorage.getItem('adminToken');
    const cacheBuster = `_t=${Date.now()}&_r=${Math.random().toString(36).substring(2, 15)}`;
    
    const response = await fetch(`/api/admin/update-credentials?${cacheBuster}`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
        'Cache-Control': 'no-cache, no-store, must-revalidate',
        'Pragma': 'no-cache'
      },
      body: JSON.stringify({
        username: username || undefined,
        password: password || undefined
      }),
      cache: 'no-store'
    });
    
    if (response.ok) {
      showStatusMessage('Login credentials updated successfully');
      document.getElementById('adminPassword').value = '';
      document.getElementById('confirmPassword').value = '';
    } else {
      const data = await response.json();
      showStatusMessage(data.message || 'Failed to update credentials', true);
    }
  } catch (err) {
    console.error('Update credentials error:', err);
    showStatusMessage('Connection error. Please try again.', true);
  }
}

// Create character counter for input or textarea
function setupCharacterCounter(element, limit) {
  // Create counter element
  const counter = document.createElement('div');
  counter.className = 'character-count';
  counter.id = element.id + '-counter';
  
  // Insert after the element
  element.parentNode.insertBefore(counter, element.nextSibling);
  
  // Initial update
  updateCharacterCounter(element, limit);
}

// Update character counter
function updateCharacterCounter(element, limit) {
  const counter = document.getElementById(element.id + '-counter');
  if (!counter) return;
  
  const length = element.value.length;
  counter.textContent = `${length}/${limit} characters`;
  
  // Update styling based on length
  counter.className = 'character-count';
  if (length > limit * 0.8 && length <= limit) {
    counter.classList.add('warning');
  } else if (length > limit) {
    counter.classList.add('error');
  }
}

// Function to remove profile image
async function removeProfileImage() {
  try {
    const token = localStorage.getItem('adminToken');
    if (!token) {
      checkAuthenticationAsync();
      return;
    }
    
    // Show a confirmation dialog
    if (!confirm('Are you sure you want to remove the profile image?')) {
      return;
    }
    
    // Show status message
    showStatusMessage('Removing profile image...');
    
    // Send request to remove image
    const response = await fetch('/api/admin/remove-image', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
        'Cache-Control': 'no-cache, no-store, must-revalidate'
      }
    });
    
    if (response.ok) {
      // Set the preview to the default SVG image
      document.getElementById('profileImagePreview').src = '/assets/Round.svg';
      
      // Reset file input
      document.getElementById('profileImage').value = '';
      
      showStatusMessage('Profile image removed successfully');
    } else {
      const data = await response.json();
      showStatusMessage(data.message || 'Failed to remove profile image', true);
    }
  } catch (err) {
    console.error('Error removing profile image:', err);
    showStatusMessage('Error removing profile image', true);
  }
}

// Function to remove background video
async function removeBackgroundVideo() {
  try {
    const token = localStorage.getItem('adminToken');
    if (!token) {
      checkAuthenticationAsync();
      return;
    }
    
    // Show a confirmation dialog
    if (!confirm('Are you sure you want to remove the background video?')) {
      return;
    }
    
    // Show status message
    showStatusMessage('Removing background video...');
    
    // Send request to remove video
    const response = await fetch('/api/admin/remove-video', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
        'Cache-Control': 'no-cache, no-store, must-revalidate'
      }
    });
    
    if (response.ok) {
      // Clear the preview
      const videoPreview = document.getElementById('backgroundVideoPreview');
      videoPreview.src = '';
      videoPreview.style.display = 'block';
      
      // Reset file input
      document.getElementById('backgroundVideo').value = '';
      
      // Show the thumbnail info again
      const thumbnailInfo = document.getElementById('videoThumbnailInfo');
      if (thumbnailInfo) {
        thumbnailInfo.style.display = 'flex';
      }
      
      showStatusMessage('Background video removed successfully');
    } else {
      const data = await response.json();
      showStatusMessage(data.message || 'Failed to remove background video', true);
    }
  } catch (err) {
    console.error('Error removing background video:', err);
    showStatusMessage('Error removing background video', true);
  }
}

// Function to remove countdown image
async function removeCountdownImage() {
  try {
    const token = localStorage.getItem('adminToken');
    if (!token) {
      checkAuthenticationAsync();
      return;
    }
    
    // Show a confirmation dialog
    if (!confirm('Are you sure you want to remove the countdown image?')) {
      return;
    }
    
    // Show status message
    showStatusMessage('Removing countdown image...');
    
    // Send request to remove image
    const response = await fetch('/api/admin/remove-countdown-image', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
        'Cache-Control': 'no-cache, no-store, must-revalidate'
      }
    });
    
    if (response.ok) {
      // Set the preview to the placeholder image
      document.getElementById('countdownImagePreview').src = '/assets/placeholder-image.png';
      
      // Reset file input
      document.getElementById('countdownImage').value = '';
      
      showStatusMessage('Countdown image removed successfully');
    } else {
      const data = await response.json();
      showStatusMessage(data.message || 'Failed to remove countdown image', true);
    }
  } catch (err) {
    console.error('Error removing countdown image:', err);
    showStatusMessage('Error removing countdown image', true);
  }
}
